/*= driver MX5060 =========================================================*/
/* Original Release: 05/03/2013                                            */
/* By: O.L.                                                                */
/* Originally written in C                                                 */
/*=========================================================================*/


#include "rs232.h"
#include "formatio.h"
#include "utility.h"
#include "ansi_c.h"
#include "mx5060.h"

#define TEMPO			0.1

#define MX5060_VDC		0
#define MX5060_VAC		1
#define MX5060_VACDC	2
#define MX5060_VLOWZ	3
#define MX5060_ADC		4
#define MX5060_AAC		5
#define MX5060_AACDC	6
#define MX5060_HZ 		7
#define MX5060_F  		8
#define MX5060_OHM		9
#define MX5060_VD 		10
#define MX5060_DEGC 	11
#define MX5060_DEGF 	12



/*= STATIC VARIABLES ======================================================*/
static int portCom;
static int mx5060_err;
static int mx5060_model;
char *mx5060_Function[4] = {"DC", "AC", "ACDC", "VLOWZ"};
char mx5060_smeasure[9];
int mx5060_typeMesure;
int mx5060_unit;


/*= UTILITY ROUTINES ======================================================*/
int mx5060_invalid_integer_range (int val, int min, int max, int err_code);
int mx5060_sendData (char *commande);
int mx5060_readData (char *buff);
double mx5060_CoeffMesure (char mult);


/*=========================================================================*/
/* Cette fonction ouvre le port srie , interroge l'appareil connect au   */
/* PC, reset l'appareil.                                                   */
/*=========================================================================*/
int mx5060_init (int serialPort, int IDQuery, int resetDevice)
{
	char buff[50];

     if (mx5060_invalid_integer_range (serialPort, 1, 255, ERROR_PARAMETER1) != 0)
        return mx5060_err;
    if (mx5060_invalid_integer_range (IDQuery, 0, 1, ERROR_PARAMETER2) != 0)
        return mx5060_err;
    if (mx5060_invalid_integer_range (resetDevice, 0, 1, ERROR_PARAMETER3) != 0)
        return mx5060_err;

	// open serial port
	OpenComConfig (serialPort, "", 4800, 0, 8, 1, 512, 512);
    
    if (rs232err == 0)
    	{
		portCom = serialPort;
	    SetComTime (portCom, 0.5);
	    SetXMode (portCom, 0);
		SetCTSMode (portCom, LWRS_HWHANDSHAKE_OFF);
		
		if (IDQuery)
			{
			mx5060_err = mx5060_sendData ("\r*IDN?\r");
			Delay (2 * TEMPO);
			mx5060_err = mx5060_readData (buff);
			if (FindPattern (buff, 0, -1, "MX5060", 0, 0) == -1)
				{
				mx5060_err = -1;
				CloseCom (portCom);
				}
			else
				{
				Scan (buff, "MX%i", &mx5060_model);

				// si appareil non calibr
				if (mx5060_model == 0)
					mx5060_model += 5060;
				}
			}

		if (resetDevice)
			mx5060_err = mx5060_sendData ("*RST\r");

		// effacer les erreurs
		mx5060_sendData ("*CLS\r");
		}
    else
		mx5060_err = rs232err;
		
	return mx5060_err;
}


/*=========================================================================*/
/* Cette fonction configure les mesures de tension de l'appareil	       */
/*=========================================================================*/
int mx5060_voltage (int function, int range, int secondDisplay, int lpass)
{
	double Range[7] = {0, 0.01, 0.1, 1, 10, 100, 700};
	char message[200];

    if (mx5060_invalid_integer_range (function, 0, 3, ERROR_PARAMETER1) != 0)
        return mx5060_err;
    if (mx5060_invalid_integer_range (range, 0, 6, ERROR_PARAMETER2) != 0)
        return mx5060_err;
    if (mx5060_invalid_integer_range (secondDisplay, 0, 5, ERROR_PARAMETER3) != 0)
        return mx5060_err;
    if (mx5060_invalid_integer_range (lpass, 0, 1, ERROR_PARAMETER4) != 0)
        return mx5060_err;

	// send command
	if (function == 3)	// function VlowZ
	{
		mx5060_err = mx5060_sendData ("FUNC \"VLOWZ\"\r");
		mx5060_typeMesure = MX5060_VLOWZ;
	}
	else
		mx5060_err = mx5060_sendData ("FUNC \"VOLT\"\r");
	
	// test commut
	mx5060_error (message);
	if (FindPattern (message, 0, -1, "-221,", 0, 0) != -1)
		mx5060_err = 301;
	
	if (mx5060_err == 0)
	{
		// Coupling
		if (function < 3)
		{
			Fmt (message, "INP:COUP %s\r", mx5060_Function[function]);
			mx5060_err = mx5060_sendData (message);
			mx5060_typeMesure = MX5060_VDC + function;
		}
	
		// Second display
		Fmt (message, "SEC %i\r", secondDisplay);
		mx5060_err = mx5060_sendData (message);
		
		// AutoRange On/Off
		Fmt (message, "RANG:AUTO %i\r", range == 0);
		mx5060_err = mx5060_sendData (message);

		// Ranging
		if (range != 0)
		{
			Fmt (message, "RANG %f\r", Range[range]);
			mx5060_err = mx5060_sendData (message);
		}
		
		// LowPass
		if (lpass > 0)
		{
			Fmt (message, "FILT 1\r");
			mx5060_err = mx5060_sendData (message);
		}
	}

	return mx5060_err;
}

/*=========================================================================*/
/* Cette fonction configure les mesures de courant de l'appareil	       */
/*=========================================================================*/
int mx5060_current (int function, int range, int secondDisplay, int lpass)
{
	double Range[6] = {0, 0.001, 0.01, 0.1, 1.0, 8.0};
	char message[200];

    if (mx5060_invalid_integer_range (function, 0, 2, ERROR_PARAMETER1) != 0)
        return mx5060_err;
    if (mx5060_invalid_integer_range (range, 0, 5, ERROR_PARAMETER2) != 0)
        return mx5060_err;
    if (mx5060_invalid_integer_range (secondDisplay, 0, 5, ERROR_PARAMETER3) != 0)
        return mx5060_err;
    if (mx5060_invalid_integer_range (lpass, 0, 1, ERROR_PARAMETER4) != 0)
        return mx5060_err;

	// send command
	mx5060_err = mx5060_sendData ("FUNC \"CURR\"\r");
	
	// test commut
	mx5060_error (message);
	if (FindPattern (message, 0, -1, "-221,", 0, 0) != -1)
		mx5060_err = 301;
	
	if (mx5060_err == 0)
	{
		// Coupling
		Fmt (message, "INP:COUP %s\r", mx5060_Function[function]);
		mx5060_err = mx5060_sendData (message);
		mx5060_typeMesure = MX5060_ADC + function;
	
		// Second display
		Fmt (message, "SEC %i\r", secondDisplay);
		mx5060_err = mx5060_sendData (message);
		
		// AutoRange On/Off
		Fmt (message, "RANG:AUTO %i\r", range == 0);
		mx5060_err = mx5060_sendData (message);

		// Ranging
		if (range != 0)
		{
			Fmt (message, "RANG %f\r", Range[range]);
			mx5060_err = mx5060_sendData (message);
		}
		
		// LowPass
		if (lpass > 0)
		{
			Fmt (message, "FILT 1\r");
			mx5060_err = mx5060_sendData (message);
		}
	}
	
	return mx5060_err;
}

/*=========================================================================*/
/* Cette fonction configure les mesures de frquence de l'appareil	       */
/*=========================================================================*/
int mx5060_frequency (int secondDisplay)
{
	char message[200];

    if (mx5060_invalid_integer_range (secondDisplay, 0, 5, ERROR_PARAMETER1) != 0)
        return mx5060_err;

	// send command
	mx5060_err = mx5060_sendData ("FUNC \"FREQ\"\r");
	mx5060_typeMesure = MX5060_HZ;
	
	// test commut
	mx5060_error (message);
	if (FindPattern (message, 0, -1, "-221,", 0, 0) != -1)
		mx5060_err = 301;
	
	if (mx5060_err == 0 && secondDisplay != 0)
	{
		// Second display
		Fmt (message, "SEC %i\r", secondDisplay);
		mx5060_err = mx5060_sendData (message);
	}
		
	return mx5060_err;
}

/*=========================================================================*/
/* Cette fonction configure les mesures de resistance et test diode de     */
/* l'appareil														       */
/*=========================================================================*/
int mx5060_resistance (int measure, int range, int secondDisplay)
{
	double Range[7] = {0, 100, 1000, 10000, 100000, 1000000, 10000000};
	char message[200];

    if (mx5060_invalid_integer_range (measure, 0, 2, ERROR_PARAMETER1) != 0)
        return mx5060_err;
    if (mx5060_invalid_integer_range (range, 0, 6, ERROR_PARAMETER2) != 0)
        return mx5060_err;
    if (mx5060_invalid_integer_range (secondDisplay, 0, 5, ERROR_PARAMETER3) != 0)
        return mx5060_err;

	// send command
	mx5060_err = mx5060_sendData ("FUNC \"RES\"\r");
	
	// test commut
	mx5060_error (message);
	if (FindPattern (message, 0, -1, "-221,", 0, 0) != -1)
		mx5060_err = 301;
	
	if (mx5060_err == 0)
	{
		if (measure == 1)
			mx5060_err = mx5060_sendData ("FUNC \"CONT\"\r");
		else if (measure == 2)
		{
			mx5060_err = mx5060_sendData ("FUNC \"DIOD\"\r");
			mx5060_typeMesure = MX5060_VD;
		}
		else
		{
			mx5060_typeMesure = MX5060_OHM;

			// Second display
			if (secondDisplay != 0)
			{
				Fmt (message, "SEC %i\r", secondDisplay);
				mx5060_err = mx5060_sendData (message);
			}
			
			// AutoRange On/Off
			Fmt (message, "RANG:AUTO %i\r", range == 0);
			mx5060_err = mx5060_sendData (message);

			// Ranging
			if (range != 0)
			{
				Fmt (message, "RANG %f\r", Range[range]);
				mx5060_err = mx5060_sendData (message);
			}
		}
	}
		
	return mx5060_err;
}

/*=========================================================================*/
/* Cette fonction configure les mesures de capacite de l'appareil		   */
/*=========================================================================*/
int mx5060_capacitance (int range, int secondDisplay)
{
	double Range[9] = {0, 1e-9, 10e-9, 100e-9, 1e-6, 10e-6, 100e-6, 1e-3, 10e-3};
	char message[200];

    if (mx5060_invalid_integer_range (range, 0, 8, ERROR_PARAMETER1) != 0)
        return mx5060_err;
    if (mx5060_invalid_integer_range (secondDisplay, 0, 5, ERROR_PARAMETER2) != 0)
        return mx5060_err;

	// send command
	mx5060_err = mx5060_sendData ("FUNC \"CAPA\"\r");
	mx5060_typeMesure = MX5060_F;
	
	// test commut
	mx5060_error (message);
	if (FindPattern (message, 0, -1, "-221,", 0, 0) != -1)
		mx5060_err = 301;
	
	if (mx5060_err == 0)
	{
		// Second display
		if (secondDisplay != 0)
		{
			Fmt (message, "SEC %i\r", secondDisplay);
			mx5060_err = mx5060_sendData (message);
		}
		
		// AutoRange On/Off
		Fmt (message, "RANG:AUTO %i\r", range == 0);
		mx5060_err = mx5060_sendData (message);

		// Ranging
		if (range != 0)
		{
			Fmt (message, "RANG %f\r", Range[range]);
			mx5060_err = mx5060_sendData (message);
		}
	}
		
	return mx5060_err;
}

/*=========================================================================*/
/* Cette fonction configure les mesures de capacite de l'appareil		   */
/*=========================================================================*/
int mx5060_temperature (int unit, int range)
{
	char *Unit[2] = {"C", "F"};
	double Range[3] = {0.0, 100, 1000};
	char message[200];

    if (mx5060_invalid_integer_range (unit, 0, 1, ERROR_PARAMETER1) != 0)
        return mx5060_err;
    if (mx5060_invalid_integer_range (range, 0, 2, ERROR_PARAMETER2) != 0)
        return mx5060_err;

	// send command
	mx5060_err = mx5060_sendData ("FUNC \"TEMP\"\r");
	
	// test commut
	mx5060_error (message);
	if (FindPattern (message, 0, -1, "-221,", 0, 0) != -1)
		mx5060_err = 301;
	
	if (mx5060_err == 0)
	{
		Fmt (message, "UNIT:TEMP %s\r", Unit[unit]);
		mx5060_err = mx5060_sendData (message);
		mx5060_typeMesure = MX5060_DEGC + unit;
		
		// AutoRange On/Off
		Fmt (message, "RANG:AUTO %i\r", range == 0);
		mx5060_err = mx5060_sendData (message);

		// Ranging
		if (range != 0)
		{
			Fmt (message, "RANG %f\r", Range[range]);
			mx5060_err = mx5060_sendData (message);
		}
	}
		
	return mx5060_err;
}

/*=========================================================================*/
/* Cette fonction configure le filtre de l'appareil			   			   */
/*=========================================================================*/
int mx5060_filter (int filter)
{
	char message[200];

    if (mx5060_invalid_integer_range (filter, 0, 1, ERROR_PARAMETER1) != 0)
        return mx5060_err;

	// send command
	Fmt (message, "FILT %i\r", filter);
	mx5060_err = mx5060_sendData (message);

	return mx5060_err;
}

/*=========================================================================*/
/* Cette fonction lit la mesure en cours							   	   */
/*=========================================================================*/
int mx5060_measure (double *measure1, int *type1)
{
	static unsigned char message[200];
	int i = 1, idxUnit, cnt;
	double mult = 1.0;
	
	memset (message, 0, 200);
	mx5060_err = mx5060_sendData ("READ?\r");
	mx5060_err = mx5060_readData (message);
	
	// test overload "OL"
	if ((cnt = FindPattern (message, 0, 8, "OL", 0, 0)) != -1 || 
		(cnt = FindPattern (message, 0, 8, "----", 0, 0)) != -1)
	{
    	mx5060_err = 303;
		Fmt (mx5060_smeasure, "  XXXXX");
		
		if (FindPattern (message, 0, 8, "----", 0, 0) == -1)
		{
			if (message[0] == '-')
			{
		    	mx5060_err = 305;
				mx5060_smeasure[0] = '-';
			}
		}
		
		idxUnit = FindPattern (message + cnt, 0, 8, " ", 0, 0) + 2;
	}
	else
	{
		// recuperer la valeur numrique
		Scan (message, "%f", measure1);

		// chercher la fin de la valeur numrique
		while (message[i] != 0)
		{
			if (message[i]>0x2C && message[i]<0x3A)
				i++;
			else
				break;
		}
	
		// corriger les caractres multiplicateurs
		idxUnit = i + 1;
		if (message[idxUnit] == 'u')
			message[idxUnit] = '';
	
		// appliquer le multiplicateur sur la valeur numrique
		mult = mx5060_CoeffMesure (message[idxUnit]);
		(*measure1) *= mult;
	
		// reconstruire la chaine alphanumrique de longueur constante
		if (i < 8)
		{
			memset (mx5060_smeasure, 0x20, 9);
		
			// ajouter le multiplicateur si il existe
			if (mult != 1.0)
				Fmt (mx5060_smeasure, "%s[i*]<%s[w*]%c", 7-i, i, message, message[i+1]);
			else
				Fmt (mx5060_smeasure, "%s[i*]<%s[w*] ", 7-i, i, message);
		
			if (mx5060_model == 5060)
				i = 0;
			else 
				i = 1;
		
			// si aucun point decimal dans la chaine de caracteres, ajouter un espace  gauche
			if (FindPattern (message, 0, 7, ".", 0, 0) == -1)
				i++;
		
			// deplacer le signe '-' si valeur ngative
			if (*measure1 < 0.0)
				 mx5060_smeasure[i++] = '-';
			else 
				mx5060_smeasure[i++] = ' ';
		
			// remplir avec des '0' les digits manquants
			while (mx5060_smeasure[i] < 0x30 && i < 6)
				mx5060_smeasure[i++] = '0';
		}
	}
	
	// detection si le commut est toujours sur la bonne position
	idxUnit += 1;
    switch (message[idxUnit])
    {
    	case 'V':	// mesure Volt
    		if (message[idxUnit+1] == 'A' && message[idxUnit+2] == 'C')
    		{
				if (message[idxUnit+3] == 'D')	
    				mx5060_unit = MX5060_VACDC;	// mesure ACDC
    			else
					mx5060_unit = MX5060_VAC;	// mesure AC
    		}
			else if (message[idxUnit+1] == 'D')
			{
				if (message[idxUnit+2] == 'C')
					mx5060_unit = MX5060_VDC;	// mesure DC
				else
					mx5060_unit = MX5060_VD;	// mesure Diode
			}
			else
				mx5060_unit = MX5060_VLOWZ;		// mesure LowZ
    		break;
		
    	case 'A':	// mesure Ampere
    		if (message[idxUnit+1] == 'A' && message[idxUnit+2] == 'C')
    		{
				if (message[idxUnit+3] == 'D')	
    				mx5060_unit = MX5060_AACDC;	// mesure ACDC
    			else
					mx5060_unit = MX5060_AAC;	// mesure AC
    		}
			else
				mx5060_unit = MX5060_ADC;		// mesure DC
    		break;
		
    	case 'H':	// mesure Hertz
    		mx5060_unit = MX5060_HZ;			// mesure frquence
    		break;
		
    	case 'F':	// mesure Farrad ou F
			mx5060_unit = MX5060_F;			// mesure capacit
    		break;
		
    	case 'o':	// mesure Ohm
    		mx5060_unit = MX5060_OHM;			// mesure rsistance
    		break;
		
    	case 0xB0:	// mesure 
    		if (message[idxUnit+1] == 'F')
				mx5060_unit = MX5060_DEGF;		// mesure F
			else
	    		mx5060_unit = MX5060_DEGC;		// mesure C
    		break;
    	}
	
	// afficher les mesures Capa, Frequence et Temperature sur 4 digits au lieu de 5
	if (mx5060_unit == MX5060_HZ || mx5060_unit == MX5060_F || 
		mx5060_unit == MX5060_DEGF || mx5060_unit == MX5060_DEGC)
	{
		mx5060_smeasure[1] = mx5060_smeasure[0];
		mx5060_smeasure[0] = ' ';
	}

	// affectation pour detection si commut modifi pendant la mesure
	*type1 = mx5060_unit;
	
	return mx5060_err;
}

/*=========================================================================*/
/* Cette function ferme le port srie                                      */
/*                                                                         */
/*=========================================================================*/
int mx5060_error (char message[])
{
	int index;
	
	mx5060_err = mx5060_sendData ("SYST:ERR?\r");
	mx5060_err = mx5060_readData (message);
	index = FindPattern (message, 0, -1, "\n", 0, 0);
	if (index != -1)
		message[index] = 0x0;
	
	return mx5060_err;
}


/*=========================================================================*/
/* Cette function ferme le port srie                                      */
/*                                                                         */
/*=========================================================================*/
int mx5060_close (void)
{

	mx5060_err = mx5060_sendData ("SYST:LOC\r");
    mx5060_err = CloseCom (portCom);
	
	return mx5060_err;
}


/*=========================================================================*/
/* cette fonction crit dans un buffer de data la chaine de commande       */
/*  envoyer                                                               */
/*=========================================================================*/
int mx5060_sendData (char *commande)
{
	FlushInQ (portCom);
	FlushOutQ (portCom);
    ComWrt (portCom, commande, strlen (commande));
    Delay (TEMPO);
    
    if (rs232err != 0)
		mx5060_err = rs232err;
		
	return mx5060_err;
}


/*=========================================================================*/
/* cette fonction lit dans un buffer la chaine de commande renvoye        */
/*                                                                         */
/*=========================================================================*/
int mx5060_readData (char *buff)
{

    ComRdTerm (portCom, buff, 50, 0x0D);

    if (rs232err != 0)
		mx5060_err = rs232err;
		
	return mx5060_err;
}

/*=========================================================================*/
/* cette fonction applique le multiplicateur  la mesure			       */
/*                                                                         */
/*=========================================================================*/
double mx5060_CoeffMesure (char mult)
	{
	double measure = 1.0;
	
	switch (mult)
		{
		case 'n' : measure = 1e-9; break;
		case '' : measure = 1e-6; break;
		case 'm' : measure = 1e-3; break;
		case 'k' : measure = 1e+3; break;
		case 'M' : measure = 1e+6; break;
		case 'G' : measure = 1e+9; break;
		}
		
	return measure;
	}

/*=========================================================================*/
/* Function: Invalid Integer Range                                         */
/* Purpose:  This function checks an integer to see if it lies between a   */
/*           minimum and maximum value.  If the value is out of range, set */
/*           the global error variable to the value err_code.  If the      */
/*           value is OK, error = 0.                                       */
/*=========================================================================*/
int mx5060_invalid_integer_range (int val, int min, int max, int err_code)
{
	if ((val < min) || (val > max))
    {
    	mx5060_err = err_code;
    	return mx5060_err;
    }
  	return 0;
}

