#include "OX9000.h"
#include <ansi_c.h>
#include <utility.h>
#include <formatio.h>
#include <visa.h>		  


/*===================================================================================*/
#define mxOX9000_REVISION  "Rev 1.0,05/2021,CVI 2012" /* Instrument driver revision  */
#define BUFFER_SIZE        512L                       /* File I/O buffer size        */
#define INVALID_SAMPLE     0xFFFF                     /* Sample validity criterion   */
#define MAX_MEM_SIZE       1610000                    /* Memory needed for file I/O  */
#define INVALID_VALUE      1E100                      /* Invalid sample default value*/
#define TMO_VALUE          1000                       /* Time out value              */
#define TMO_TRANSFER       10000                      /* Time out value              */
#define HEADER_SIZE        200                        /* Size of binary data header  */
#define HEADER_FILE        20                         /* Size of binary file header  */
/*===================================================================================*/

/*=mxOX9000==================================================================*/
/* LabWindows/CVI 2021 Instrument Driver (VISA I/O)                          */
/* Original Release: May 2021                                                */
/* By:Metrix, Annecy, France                                                 */
/*                                                                           */                           
/* Modification History: None                                                */
/*                                                                           */
/*===========================================================================*/
 
/*****************************************************************************/
/*= INSTRUMENT-DEPENDENT COMMAND ARRAYS =====================================*/
/*****************************************************************************/
/*-Vertical command arrays---------------------------------------------------*/
static ViString cmdVertCoupling[] = {"AC","DC","GRO"};       
static ViString cmdVoltDiv[] = {"20mV","40mV","80mV", "160mV","400mV","800mV",
								"1.6V","4V","8V", "16V","40V","80V", "160V","400V","800V","1.6kV"}; 
						
static ViReal64 cmdOffsetMax[] = {0.025, 0.5, 0.1, 0.2, 0.5, 1.0, 2.0, 5.0, 10.0, 20.0,
								  50.0, 100.0, 200.0, 500.0, 1000.0, 2000.0}; 
static ViReal64 cmdBandwith[] = {0.0, 15e6, 1.5e6, 5e3}; 
static ViString cmdInpSelect[]  = {"CHAN","MEM"};   

/*-Horizontal command arrays-------------------------------------------------*/
static ViString cmdTimeDiv[] = {"1E-9", "2E-9", "5E-9", "10E-9", "25E-9", "50E-9", "100E-9", "200E-9", "500E-9",
								"1E-6", "2E-6", "4E-6", "10E-6", "20E-6", "50E-6", "100E-6", "200E-6", "500E-6",
								"1E-3", "2E-3", "5E-3", "10E-3", "20E-3", "40E-3", "100E-3", "200E-3", "500E-3",
								"1", "2" ,"5", "10", "20", "50", "100", "200"};
                        
static ViString cmdDispTrace[] = {"A", "XY"}; 
static ViString cmdcalcfft[] = {"RECT", "HAMM", "HANN", "BLAC"};
static ViString cmdScale[] = {"LIN", "LOG"};

/*-Trigger command arrays---------------------------------------------------*/
static ViString cmdTrigCoupling[] = {"AC","DC","ON"};       
static ViString cmdSlope[] = {"NEG", "POS"};
static ViString cmdSource[]= {"INT1", "INT2", "INT3", "INT4"}; 
static ViString cmdType[] = {"0","1","0"};  
static ViString CmdPulsTyp[] = {"INF", "SUP", "INS", "OUT"};
/*-Display command arrays----------------------------------------------------*/
static ViString cmddevice[]={"HOME", "SCOP", "ANALYS", "LOG" ,"MULT"};
static ViString cmdDisplayScreen[] = {"0", "15", "30", "60"};
static ViString cmdSystemStandby[] = {"30", "60", "240", "1440"};
static ViString cmdMaxminAcq[] = {"ENV", "NORM"};
static ViString cmdMode[] = {"NORM", "ENV", "ALL", "ACC"};
static ViString cmdAverage[] = {"0", "2", "4", "16", "64"};       
/*-Read command arrays-------------------------------------------------------*/
static ViString cmdReadMeasure[] = {"MIN", "MAX", "PTP", "LOW", "HIGH", "AMPL", "AC", "AC", "VOLT", "SUM",
									"RTIME", "FTIME", "PWID","NWID", "PER", "FREQ", "PDUT", "PUL:COUN",
									"RISE:OVER", "FALL:OVER", "PHAS"};
static ViString cmdMeasRMS[] = {"INT", "CYC"};
static ViReal64 cmdTimeBase[] = {1E-9, 2E-9, 5E-9, 10E-9, 25E-9, 50E-9, 100E-9, 200E-9, 500E-9,
								 1E-6, 2E-6, 4E-6, 10E-6, 20E-6, 50E-6, 100E-6, 200E-6, 500E-6, 
								 1E-3, 2E-3, 5E-3, 10E-3, 20E-3, 40E-3, 100E-3, 200E-3, 500E-3,
								 1.0, 2.0, 5.0, 10.0, 20.0, 50.0, 100.0, 200.0};
static ViReal64 cmdVolt[] = {2.5E-3,5E-3,10E-3,20E-3,50E-3,100E-3,200E-3,500E-3,1.0,
							2.0,5.0,10.0,20.0,50.0,100.0,200.0};
/*-Meter command arrays-------------------------------------------------------*/
static ViString MulCoup[] = {"DC", "AC", "ACDC"};
static ViReal64 MulBand[] = {0.0, 5e+3, 625.0};
static ViString CmdMultMeas[] = {"VOLT", "RES", "CONT", "CAPA", "DIOD", "PT100"};
static ViString CmdMultVolt[] = {"800E-3", "8.00", "80.0", "800"};
static ViString CmdMultOhm[]  = {"80.0", "800", "8.0E+3", "80.0E+3", "800E+3", "8.0E+6", "32.0E+6"};
static ViString CmdMulCapa[]  = {"5.00E-9", "50E-9", "500E-9", "5.00E-6", "50E-6", "500E-6", "5.00E-3"};
                                     
/*****************************************************************************/
/*= INSTRUMENT-DEPENDENT STATUS/RANGE STRUCTURE  ============================*/
/*****************************************************************************/
/*  mxOX9000_stringValPair is used in the  mxOX9000_errorMessage function    */
/*===========================================================================*/
typedef struct   mxOX9000_stringValPair
{
   ViStatus stringVal;
   ViPString stringName;
}   mxOX9000_tStringValPair;
	 
struct mxOX9000_statusDataRanges {
    ViInt16 triggerMode;
    ViInt16 val2;
    ViInt16 val3;
    ViChar instrDriverRevision[256];
    ViInt16 instrMode;
};
typedef struct mxOX9000_statusDataRanges * mxOX9000_instrRange;

ViInt16 TrigMode = 0 ;  
/*****************************************************************************/
/*= UTILITY ROUTINE DECLARATIONS (Non-Exportable Functions) =================*/
/*****************************************************************************/
ViBoolean mxOX9000_invalidViBooleanRange (ViBoolean val);
ViBoolean mxOX9000_invalidViInt16Range (ViInt16 val, ViInt16 min, ViInt16 max);
ViBoolean mxOX9000_invalidViInt32Range (ViInt32 val, ViInt32 min, ViInt32 max);
ViBoolean mxOX9000_invalidViUInt16Range (ViUInt16 val, ViUInt16 min, ViUInt16 max);
ViBoolean mxOX9000_invalidViUInt32Range (ViUInt32 val, ViUInt32 min, ViUInt32 max);
ViBoolean mxOX9000_invalidViReal32Range (ViReal32 val, ViReal32 min, ViReal32 max);
ViBoolean mxOX9000_invalidViReal64Range (ViReal64 val, ViReal64 min, ViReal64 max);
ViStatus  mxOX9000_makeDataReadable (ViByte tabByte[], ViReal64 _VI_FAR waveformArray[], ViPInt32 numberOfSamples,
                                	 ViPReal64 XStart, ViPReal64 XIncrement);
ViBoolean mxOX9000_invalidFilename (ViString filename, ViInt16 drive, ViString type);
ViStatus  mxOX9000_instrStatus (ViSession instrSession);
ViStatus  mxOX9000_initCleanUp (ViSession openRMSession, ViPSession openInstrSession, ViStatus currentStatus);
ViStatus  mxOX9000_defaultInstrSetup (ViSession openInstrSession);

/*****************************************************************************/
/*====== USER-CALLABLE FUNCTIONS (Exportable Functions) =====================*/
/*****************************************************************************/

/*===========================================================================*/
/* Function:	Initialize                                                   */
/* Purpose:		This function opens the instrument, queries the instrument   */
/*				for its ID, and initializes the instrument to a known state  */
/*===========================================================================*/
ViStatus _VI_FUNC  mxOX9000_init (ViRsrc resourceName, ViBoolean IDQuery, ViPSession instrSession)
{
	ViStatus  mxOX9000_status = VI_SUCCESS;
	ViSession rmSession = 0;
	ViUInt32 retCnt = 0;
	ViByte rdBuffer[BUFFER_SIZE];
	ViByte model[2];
	ViUInt16 interface;
	ViFindList Find_Handle = 0;
	char InstrumentDescriptor[300];
    
	/*- Check input parameter ranges ----------------------------------------*/
	if ( mxOX9000_invalidViBooleanRange (IDQuery))
		return VI_ERROR_PARAMETER2;

	/*- Open instrument session ---------------------------------------------*/
	if (( mxOX9000_status = viOpenDefaultRM (&rmSession)) < 0)
		return  mxOX9000_status;

	if ( FindPattern (resourceName, 0, -1, "USB", 0, 0) != -1)
	{
		if (viFindRsrc (rmSession, "ASRL[0-9]*::?*INSTR", &Find_Handle, &retCnt, InstrumentDescriptor ) == VI_SUCCESS)
		{   //-- Recherche de tous les instruments USB-serial sur le PC,
			//   Expression = "ASRL[0-9]*::?*INSTR"   => recherche de priphrique Serial
			do
			{	
				//-- Scrupte les diffrents descripteurs d'instrument
				mxOX9000_status = viOpen(rmSession, InstrumentDescriptor, VI_NULL, VI_NULL, instrSession);
				if (mxOX9000_status == VI_SUCCESS)
				{	
					// Timeout de 100ms pour scanner rapidement les diffrents ports COM
					viSetAttribute (*instrSession, VI_ATTR_TMO_VALUE, 100);
					// Rgler le caractre de fin de chaine
					viSetAttribute (*instrSession, VI_ATTR_TERMCHAR, 0x0D);
					// tester la rponse de l'appareil
					viWrite (*instrSession, "\r*IDN?\r", 7, &retCnt);
					viRead (*instrSession, rdBuffer, 50, &retCnt);
					if (retCnt != 0)
					{
						if (rdBuffer[0] == 0)
							rdBuffer[0] = 0x20;

						// rponse lue si Scopix4
						if (FindPattern (rdBuffer, 0, -1, "OX9", 0, 0) != -1)
							break;
					}
					else
						viClose (*instrSession);
				}
				// recherhce de la ressource Serial suivante
				mxOX9000_status = viFindNext (Find_Handle, InstrumentDescriptor);
			}
			while (mxOX9000_status == VI_SUCCESS);
			
			// erreur si aucun Scopix 4 n'a rpondu
			if (mxOX9000_status != VI_SUCCESS)
			{
				viClose (rmSession);
				return -1;
			}
		}
	}
	else
	{
		if ((mxOX9000_status = viOpen (rmSession, resourceName, VI_NULL, VI_NULL, instrSession)) < 0) 
		{
			viClose (rmSession);
			return mxOX9000_status;
		}	
	}
	
	/*- Configure VISA Formatted I/O ----------------------------------------*/
	if ((mxOX9000_status = viSetAttribute (*instrSession, VI_ATTR_TERMCHAR, 0x0D)) < 0)
		return mxOX9000_initCleanUp (rmSession, instrSession,  mxOX9000_status);
	if ((mxOX9000_status = viSetAttribute (*instrSession, VI_ATTR_TERMCHAR_EN, VI_FALSE)) < 0)
		return mxOX9000_initCleanUp (rmSession, instrSession,  mxOX9000_status);
	if ((mxOX9000_status = viSetAttribute (*instrSession, VI_ATTR_TMO_VALUE, TMO_VALUE)) < 0)
		return mxOX9000_initCleanUp (rmSession, instrSession,  mxOX9000_status);
	if ((mxOX9000_status = viSetBuf (*instrSession, VI_READ_BUF|VI_WRITE_BUF, 4000)) < 0)
		return mxOX9000_initCleanUp (rmSession, instrSession,  mxOX9000_status);
	if ((mxOX9000_status = viSetAttribute (*instrSession, VI_ATTR_WR_BUF_OPER_MODE, VI_FLUSH_ON_ACCESS)) < 0)
		return mxOX9000_initCleanUp (rmSession, instrSession,  mxOX9000_status);
	if ((mxOX9000_status = viSetAttribute (*instrSession, VI_ATTR_RD_BUF_OPER_MODE, VI_FLUSH_ON_ACCESS)) < 0)
		return mxOX9000_initCleanUp (rmSession, instrSession,  mxOX9000_status);
	if ((mxOX9000_status = viSetAttribute (*instrSession, VI_ATTR_SUPPRESS_END_EN, VI_FALSE)) < 0)
		return mxOX9000_initCleanUp (rmSession, instrSession,  mxOX9000_status);

	/*- Identification Query ------------------------------------------------*/
	if (IDQuery) 
	{
		if ((mxOX9000_status = viWrite (*instrSession, "*IDN?\r", 6, &retCnt)) < 0)
			return mxOX9000_initCleanUp (rmSession, instrSession,  mxOX9000_status);
		if ((mxOX9000_status = viRead (*instrSession, rdBuffer, BUFFER_SIZE, &retCnt)) < 0)
			return mxOX9000_status;
		if (FindPattern (rdBuffer, 0, -1, "OX9", 1, 0) == -1)
			return mxOX9000_initCleanUp (rmSession, instrSession, VI_ERROR_FAIL_ID_QUERY);
	}

	/*- Send Default Instrument Setup ---------------------------------*/
	if ((mxOX9000_status =  mxOX9000_defaultInstrSetup (*instrSession)) < 0)
		return mxOX9000_initCleanUp (rmSession, instrSession,  mxOX9000_status);
	
	if ((mxOX9000_status = viWrite (*instrSession, "FORM INT;FORM:DINT 1\r", 21, &retCnt)) < 0)
		return mxOX9000_initCleanUp (rmSession, instrSession,  mxOX9000_status);
	
	if ((mxOX9000_status = viWrite (*instrSession, "DEV:MOD SCOP\r", 13, &retCnt)) < 0)
		return mxOX9000_initCleanUp (rmSession, instrSession,  mxOX9000_status);

	/*-Check instrument status & Return from function------------------------*/
	if ((mxOX9000_status =  mxOX9000_instrStatus(*instrSession)) < 0)
		return mxOX9000_status;

	return  mxOX9000_status;
}

/*===========================================================================*/
/* Function: Example                                                         */
/* Purpose:  This function is an example of this driver's functions use.     */
/*===========================================================================*/
ViStatus _VI_FUNC mxOX9000_example (ViSession instrSession, ViInt16 channel, ViInt16 timebase, ViInt16 voltageRange,
									ViInt16 coupling, ViReal64 _VI_FAR data[], ViPInt32 numberOfSamples, 
									ViPReal64 XStart, ViPReal64 XIncrement)
{
	ViStatus mxOX9000_status = VI_SUCCESS;
    
	/*- Check input parameter ranges ----------------------------------------*/    
	if (mxOX9000_invalidViInt16Range (channel, 0, 3))
		return VI_ERROR_PARAMETER2;
	if (mxOX9000_invalidViInt16Range (timebase, 0, 33))
		return VI_ERROR_PARAMETER3;
	if (mxOX9000_invalidViInt16Range (voltageRange, 0, 17))
		return VI_ERROR_PARAMETER4;
	if (mxOX9000_invalidViInt16Range (coupling, 0, 2))
		return VI_ERROR_PARAMETER5;
    
	/*- Configuring vertical settings----------------------------------------*/
	if((mxOX9000_status = mxOX9000_verticalSetup (instrSession, channel + 1, VI_ON, coupling, 1, 
						voltageRange, 0.0, VI_ON)) < 0)
		return mxOX9000_status;
        
	/*- Setting timebase ----------------------------------------------------*/
	if((mxOX9000_status = mxOX9000_horizontalSetup (instrSession, timebase, -250.0)) < 0)
		return mxOX9000_status;

	/*- Setting trigger mode to AUTO-----------------------------------------*/
	/*- mxOX9000 starts acquiring data immediately after it is triggered-------*/
	if((mxOX9000_status = mxOX9000_setGeneralTrigger (instrSession, channel, VI_ON, 0, VI_OFF, 1, 0.0, 40e-9)) < 0)
		return mxOX9000_status;
        
	/*- Sending single-shot trigger in wait mode-----------------------------*/
	/*- This function will return as long as the acquisition is running------*/
	if((mxOX9000_status = mxOX9000_runAcquisition (instrSession, VI_ON)) < 0)
		return mxOX9000_status;
        
	/*- Reads trace from the instrument--------------------------------------*/ 
	if((mxOX9000_status = mxOX9000_readChToArr (instrSession, channel, data, numberOfSamples, XStart, XIncrement)) < 0)
		return mxOX9000_status; 
                   
	return mxOX9000_status;
}
 
/*===========================================================================*/
/* Function: Configure mxOX9000 Vertical Settings                            */
/* Purpose:  This function configures the vertical settings for a specified  */
/*           channel                                                         */
/*===========================================================================*/
ViStatus _VI_FUNC mxOX9000_verticalSetup (ViSession instrSession, ViInt16 input, ViBoolean state, ViInt16 coupling,
										  ViInt16 probe, ViInt16 sense, ViReal64 offset, ViInt16 band)
{
	ViStatus mxOX9000_status = VI_SUCCESS;
	ViByte wrBuffer[BUFFER_SIZE];
	ViInt32 retCnt;

	/*-Check input parameter ranges------------------------------------------*/
	if (mxOX9000_invalidViInt16Range (input, 1, 4))
		return VI_ERROR_PARAMETER2;
	if (mxOX9000_invalidViBooleanRange (state))
		return VI_ERROR_PARAMETER3;
	if (mxOX9000_invalidViInt16Range (coupling, 0, 2))
		return VI_ERROR_PARAMETER4;
	if (probe != 1 && probe != 10 && probe != 100)
		return VI_ERROR_PARAMETER5;
	if (mxOX9000_invalidViInt16Range (sense, 0, 15))
		return VI_ERROR_PARAMETER6;
	if (mxOX9000_invalidViReal64Range (offset, -cmdOffsetMax[sense], cmdOffsetMax[sense]))
		return VI_ERROR_PARAMETER7;
	if (mxOX9000_invalidViInt16Range (band, 0, 3))
		return VI_ERROR_PARAMETER8;
    
	/*-Send vertical config command------------------------------------------*/
	Fmt (wrBuffer, "DISP:TRAC:Y:PDIV%i[b2] 1\r", input);
	if ((mxOX9000_status = viWrite (instrSession, wrBuffer, strlen (wrBuffer), &retCnt)) < 0)
		return mxOX9000_status; 
	Fmt (wrBuffer, "DISP:TRAC:STAT%i[b2] %i[b2]\r", input, state);
	if ((mxOX9000_status = viWrite (instrSession, wrBuffer, strlen (wrBuffer), &retCnt)) < 0)
		return mxOX9000_status; 
	Fmt (wrBuffer, "INP%i[b2]:COUP %s\r", input, cmdVertCoupling[coupling]);
	if ((mxOX9000_status = viWrite (instrSession, wrBuffer, strlen (wrBuffer), &retCnt)) < 0)
		return mxOX9000_status; 
	Fmt (wrBuffer, "BAND%i[b2]:RES %f\r", input, cmdBandwith[band]);
	if ((mxOX9000_status = viWrite (instrSession, wrBuffer, strlen (wrBuffer), &retCnt)) < 0)
		return mxOX9000_status; 
	Fmt (wrBuffer, "VOLT%i[b2]:RANG:PTP %s\r", input, cmdVoltDiv[sense]);
	if ((mxOX9000_status = viWrite (instrSession, wrBuffer, strlen (wrBuffer), &retCnt)) < 0)
		return mxOX9000_status; 
	Fmt (wrBuffer, "VOLT%i[b2]:RANG:OFFS %f\r", input, offset);
	if ((mxOX9000_status = viWrite (instrSession, wrBuffer, strlen (wrBuffer), &retCnt)) < 0)
		return mxOX9000_status; 
	Fmt (wrBuffer, "DISP:TRAC:Y:PDIV%i[b2] %i[b2]\r", input, probe);
	if ((mxOX9000_status = viWrite (instrSession, wrBuffer, strlen (wrBuffer), &retCnt)) < 0)
		return mxOX9000_status; 
	
	/*-Check instrument status & Return from function------------------------*/
	if ((mxOX9000_status = mxOX9000_instrStatus(instrSession)) < 0)
		return mxOX9000_status;
        
	return mxOX9000_status;
}

/*===========================================================================*/
/* Function: Configure mxOX9000 Horizontal Settings                          */
/* Purpose:  This function configures the horizontal settings of the         */
/*           instrument                                                      */
/*===========================================================================*/
ViStatus _VI_FUNC mxOX9000_horizontalSetup (ViSession instrSession, ViInt16 range, ViReal64 pretrg) 
{
	ViStatus mxOX9000_status = VI_SUCCESS;
	ViByte pretrgBuf[BUFFER_SIZE];
	ViInt32 retCntptr;
    
	/*-Check input parameter ranges------------------------------------------*/
	if (mxOX9000_invalidViInt16Range (range, 0, 34))
		return VI_ERROR_PARAMETER2;
	if (mxOX9000_invalidViReal64Range (pretrg, -4640.0, 0))
		return VI_ERROR_PARAMETER3;
      
	/*----------------------- time bases ------------------------------------*/
	Fmt (pretrgBuf, "DISP:TRAC:X:PDIV %s\r", cmdTimeDiv[range]);
	if ((mxOX9000_status = viWrite (instrSession, pretrgBuf, strlen (pretrgBuf), &retCntptr)) < 0)
		return mxOX9000_status; 
        
	/*----------------  pretrig --------------------------------------------*/
	if (range < 6)
		Fmt (pretrgBuf, "SWE:OFFS:TIME %f\r", cmdTimeBase[6] * pretrg /  500.0);
	else
		Fmt (pretrgBuf, "SWE:OFFS:TIME %f\r", cmdTimeBase[range] * pretrg /  500.0);
	
	if ((mxOX9000_status = viWrite (instrSession, pretrgBuf, strlen (pretrgBuf), &retCntptr)) < 0)
		return mxOX9000_status; 

	/*-Check instrument status & Return from function------------------------*/
	if ((mxOX9000_status = mxOX9000_instrStatus(instrSession)) < 0)
		return mxOX9000_status;

	return mxOX9000_status;
}

/*===========================================================================*/
/* Function: Configure mxOX9000 Horizontal Display                           */
/* Purpose:  This function configures the horizontal display                 */
/*===========================================================================*/
ViStatus _VI_FUNC mxOX9000_horizontalDisplay (ViSession instrSession, ViInt16 trace, ViInt16 x, ViInt16 y)
{
	ViStatus mxOX9000_status = VI_SUCCESS;
	ViByte wrBuffer[BUFFER_SIZE];
	ViInt32 retCnt;

	/*-Check input parameter ranges------------------------------------------*/
	if (mxOX9000_invalidViInt16Range (trace, 0, 2))
		return VI_ERROR_PARAMETER2;
	if (mxOX9000_invalidViInt16Range (x, 1, 4))
		return VI_ERROR_PARAMETER3;
	if (mxOX9000_invalidViInt16Range (y, 1, 4))
		return VI_ERROR_PARAMETER4;
   
	if (trace == 1)
		Fmt (wrBuffer, "DISP:TRAC:FORM %s;XY:XDEF INT%i[b2];YDEF INT%i[b2]\r", cmdDispTrace[trace], x, y);
	else
		Fmt (wrBuffer, "DISP:TRAC:FORM %s\r", cmdDispTrace[trace]);
	
	if ((mxOX9000_status = viWrite (instrSession,wrBuffer, strlen (wrBuffer), &retCnt)) < 0)
		return mxOX9000_status;
        
	/*-Check instrument status & Return from function------------------------*/
	if ((mxOX9000_status = mxOX9000_instrStatus (instrSession)) < 0)
		return mxOX9000_status;
     
	return mxOX9000_status;
}

/*===========================================================================*/
/* Function: FFT                                                             */
/* Purpose:  This function configures the FFT                                */
/*===========================================================================*/
ViStatus _VI_FUNC mxOX9000_FFT (ViSession instrSession, ViBoolean fftvalid, ViInt16 calculate, ViBoolean scale)
{
	ViStatus mxOX9000_status = VI_SUCCESS;
	ViByte wrBuffer[BUFFER_SIZE];     
	ViInt32 retCnt;

	/*-Check input parameter ranges------------------------------------------*/
	if (mxOX9000_invalidViBooleanRange (fftvalid))
		return VI_ERROR_PARAMETER2;
	if (mxOX9000_invalidViInt16Range (calculate, 0, 5))
		return VI_ERROR_PARAMETER3;
	if (mxOX9000_invalidViBooleanRange (scale))
		return VI_ERROR_PARAMETER4;
    
	/*-Set ON-OFF FFT, Calculate and scale parameters----------------*/ 
	Fmt (wrBuffer, "CALC:TRAN:FREQ %i[b2]\r", fftvalid);
	if ((mxOX9000_status = viWrite (instrSession, wrBuffer, strlen (wrBuffer), &retCnt)) < 0)
		return mxOX9000_status;
	Fmt (wrBuffer, "CALC:TRAN:FREQ:WIND %s\r", cmdcalcfft[calculate]);
	if ((mxOX9000_status = viWrite (instrSession, wrBuffer, strlen (wrBuffer), &retCnt)) < 0)
		return mxOX9000_status;
	Fmt (wrBuffer, "DISP:TRAC:Y:SPAC %s\r", cmdScale[scale]);
	if ((mxOX9000_status = viWrite (instrSession, wrBuffer, strlen (wrBuffer), &retCnt)) < 0)
		return mxOX9000_status; 
    
	/*-Check instrument status & Return from function------------------------*/
	if ((mxOX9000_status = mxOX9000_instrStatus(instrSession)) < 0)
		return mxOX9000_status;   
    
	return mxOX9000_status;
}

/*===========================================================================*/
/* Function: General Trigger                                                 */
/* Purpose:  This function sets general parameters for trigger               */
/*===========================================================================*/
ViStatus _VI_FUNC mxOX9000_setGeneralTrigger (ViSession instrSession, ViInt16 channel, ViBoolean slope, ViInt16 coupling,
											  ViInt16 noiseReject, ViInt16 type, ViReal64 level, ViReal64 hold)
{
	ViStatus mxOX9000_status = VI_SUCCESS;
	ViByte wrBuffer[BUFFER_SIZE];
	ViReal64 convBuf;
	ViInt32 retCnt;
	ViByte runStat = 1;
	/*-Check input parameter ranges------------------------------------------*/
	if (mxOX9000_invalidViInt16Range (channel, 0, 3))
		return VI_ERROR_PARAMETER2;
	if (mxOX9000_invalidViBooleanRange (slope))
		return VI_ERROR_PARAMETER3;
	if (mxOX9000_invalidViInt16Range (coupling, 0, 3))
		return VI_ERROR_PARAMETER4;
	if (mxOX9000_invalidViInt16Range (type, 0, 2))
		return VI_ERROR_PARAMETER7;
	if (mxOX9000_invalidViInt16Range (noiseReject, 0, 3))
		return VI_ERROR_PARAMETER5;
	if (mxOX9000_invalidViReal64Range (level, -1600, 1600))
		return VI_ERROR_PARAMETER6;
	if (mxOX9000_invalidViReal64Range (hold, 64E-9, 15.0))
		return VI_ERROR_INSTR_PARAMETER9;
        
	/*-Set slope,coupling,level,holdoff and filter parameters----------------*/
	if ((mxOX9000_status = viPrintf (instrSession, "TRIG:SOUR INT%hd\r", channel+1)) < 0)
		return mxOX9000_status;
	if ((mxOX9000_status = viPrintf (instrSession, "TRIG:SEQ1:SLOP %s\r", cmdSlope[slope])) < 0)
		return mxOX9000_status;
	if ((mxOX9000_status = viPrintf (instrSession, "TRIG:LEV %lf\r", level)) < 0)
		return mxOX9000_status;
	Fmt (wrBuffer,"TRIG:HOLD %f\r",hold);
	if ((mxOX9000_status = viWrite (instrSession, wrBuffer, strlen (wrBuffer), &retCnt)) < 0)
		return mxOX9000_status; 
	Fmt (wrBuffer, "TRIG:HYST %i\r", noiseReject);
	if ((mxOX9000_status = viWrite (instrSession, wrBuffer, strlen (wrBuffer), &retCnt)) < 0)
		return mxOX9000_status;
        
	switch (coupling)
	{
		case 0 :
				Fmt (wrBuffer, "TRIG:COUP %s\r", cmdTrigCoupling[coupling]);
				if ((mxOX9000_status = viWrite (instrSession, wrBuffer, strlen (wrBuffer), &retCnt)) < 0)
					return mxOX9000_status;
				break;
		case 1 :
				Fmt (wrBuffer, "TRIG:COUP %s\r", cmdTrigCoupling[coupling]);
				if ((mxOX9000_status = viWrite (instrSession, wrBuffer, strlen (wrBuffer), &retCnt)) < 0)
					return mxOX9000_status;
				break;	
		case 2 :
				Fmt (wrBuffer, "TRIG:FILT:HPAS 1\r");
				if ((mxOX9000_status = viWrite (instrSession, wrBuffer, strlen (wrBuffer), &retCnt)) < 0)
					return mxOX9000_status;
				break;	
		case 3 :
				Fmt (wrBuffer, "TRIG:FILT:LPAS 1\r");
				if ((mxOX9000_status = viWrite (instrSession, wrBuffer, strlen (wrBuffer), &retCnt)) < 0)
					return mxOX9000_status;
				break;	
	}

	if ((mxOX9000_status = mxOX9000_instrStatus(instrSession)) < 0)
		return mxOX9000_status;

	/*-Set trigger type and mode parameters----------------*/
	switch (type)
	{
		case 0 :
		TrigMode = 0;
		Fmt (wrBuffer, "TRIG:SEQ1:ATRIG %s\r", cmdType[type]);
		if ((mxOX9000_status = viWrite (instrSession, wrBuffer, strlen (wrBuffer), &retCnt)) < 0)
			return mxOX9000_status;
		Fmt (wrBuffer, "INIT:CONT:NAME EDGE,1\r");
		if ((mxOX9000_status = viWrite (instrSession, wrBuffer, strlen (wrBuffer), &retCnt)) < 0)
			return mxOX9000_status;
		break;
		
		case 1 :
		TrigMode = 1;
		Fmt (wrBuffer, "TRIG:SEQ1:ATRIG %s\r", cmdType[type]);
		if ((mxOX9000_status = viWrite (instrSession, wrBuffer, strlen (wrBuffer), &retCnt)) < 0)
			return mxOX9000_status;
		break;

		case 2 :
		Fmt (wrBuffer, "INIT:NAME EDGE\r");
		if ((mxOX9000_status = viWrite (instrSession, wrBuffer, strlen (wrBuffer), &retCnt)) < 0)
			return mxOX9000_status; 
		TrigMode = 2;
		break;
	}

	/*-Check instrument status & Return from function------------------------*/
	if ((mxOX9000_status = mxOX9000_instrStatus(instrSession)) < 0)
		return mxOX9000_status;

	return mxOX9000_status;
}

/*===========================================================================*/
/* Function: PULSE Trigger                                                   */
/* Purpose:  This function sets Pulsate trigger type                         */
/*===========================================================================*/
ViStatus _VI_FUNC mxOX9000_setPulsTrigger (ViSession instrSession, ViInt16 pulsType, ViReal64 T1, ViReal64 T2)
{                                
	ViStatus mxOX9000_status = VI_SUCCESS;
	ViReal64 convBuf;
	ViByte rdBuffer[BUFFER_SIZE];
	ViInt32 retCnt;
	ViBoolean runStat;

	/*-Check input parameter ranges------------------------------------------*/
	if (mxOX9000_invalidViInt16Range (pulsType, 0, 3))
		return VI_ERROR_PARAMETER2;
	if (mxOX9000_invalidViReal64Range (T1, 16E-9, 15.0))
		return VI_ERROR_PARAMETER3;
	if (mxOX9000_invalidViReal64Range (T2, 16E-9, 15.0))
		return VI_ERROR_PARAMETER4;

   	if(TrigMode == 0)
	{
		Fmt (rdBuffer ,"TRIG:RUN:STAT?\r");
		if ((mxOX9000_status = viWrite (instrSession,rdBuffer,strlen (rdBuffer), &retCnt)) < 0)
			return mxOX9000_status;
		if ((mxOX9000_status = viScanf (instrSession,"%hd", &runStat)) < 0)
			return mxOX9000_status;
		Fmt (rdBuffer, "INIT:CONT:NAME PUL,%i[b1]\r", runStat);
		if ((mxOX9000_status = viWrite (instrSession, rdBuffer, strlen (rdBuffer), &retCnt)) < 0)
			return mxOX9000_status;
	}
	 else if(TrigMode == 2)
 	{
		Fmt (rdBuffer, "INIT:NAME PUL\r");
		if ((mxOX9000_status = viWrite (instrSession, rdBuffer, strlen (rdBuffer), &retCnt)) < 0)
			return mxOX9000_status; 
 	}
    
	/*-Set source and levels for edge trigger----------------------------*/
	if ((mxOX9000_status = viPrintf (instrSession, "TRIG:TYP %s\r", CmdPulsTyp[pulsType])) < 0)
		return mxOX9000_status;
	Fmt (rdBuffer,"TRIG:SEQ2:DEL %f\r", T1);
	if ((mxOX9000_status = viWrite (instrSession, rdBuffer, strlen (rdBuffer), &retCnt)) < 0)
		return mxOX9000_status; 
	Fmt (rdBuffer,"TRIG:SEQ2:DELD %f\r", T2);
	if ((mxOX9000_status = viWrite (instrSession, rdBuffer, strlen (rdBuffer), &retCnt)) < 0)
		return mxOX9000_status; 

	/*-Check instrument status & Return from function------------------------*/
	if ((mxOX9000_status = mxOX9000_instrStatus(instrSession)) < 0)
		return mxOX9000_status;

	return mxOX9000_status;
}

/*===========================================================================*/
/* Function: Delayed Trigger                                                 */
/* Purpose:  This function sets delayed trigger type                         */
/*===========================================================================*/
ViStatus _VI_FUNC mxOX9000_setDelayedTrigger (ViSession instrSession, ViInt16 auxi_source, ViBoolean edge,
											  ViInt16 coupling, ViInt16 noiseReject, ViReal64 level, ViReal64 delay)
{                                
	ViStatus mxOX9000_status = VI_SUCCESS;
	ViByte wrBuffer[BUFFER_SIZE];
	ViInt32 retCnt;
	ViByte runStat;

    /*-Check input parameter ranges------------------------------------------*/
    if (mxOX9000_invalidViInt16Range (auxi_source, 1, 4))
        return VI_ERROR_PARAMETER2;
	if (mxOX9000_invalidViBooleanRange (edge))
		return VI_ERROR_PARAMETER3;
	if (mxOX9000_invalidViInt16Range (coupling, 0, 3))
		return VI_ERROR_PARAMETER4;
	if (mxOX9000_invalidViInt16Range (noiseReject, 0, 3))
		return VI_ERROR_PARAMETER5;
    if (mxOX9000_invalidViReal64Range (level, -800, 800))
        return VI_ERROR_PARAMETER5;
    if (mxOX9000_invalidViReal64Range (delay, 48.0E-9, 20.0))
        return VI_ERROR_PARAMETER6;
        
    /*-Set delay parameters--------------------------------------------------*/
	if ((mxOX9000_status = viPrintf (instrSession,"ARM:SOUR INT%hd\r", auxi_source)) < 0)
		return mxOX9000_status;
	if ((mxOX9000_status = viPrintf (instrSession,"ARM:SLOP %s\r", cmdSlope[edge])) < 0)
		return mxOX9000_status;
	if ((mxOX9000_status = viPrintf (instrSession,"ARM:LEV %lf\r", level)) < 0)
		return mxOX9000_status;
	Fmt (wrBuffer,"TRIG:SEQ3:DEL %f\r",delay);
	if ((mxOX9000_status = viWrite (instrSession, wrBuffer, strlen (wrBuffer), &retCnt)) < 0)
		return mxOX9000_status; 
	switch (coupling)
	{
		case 0 :
				Fmt (wrBuffer, "ARM:COUP %s\r", cmdTrigCoupling[coupling]);
				if ((mxOX9000_status = viWrite (instrSession, wrBuffer, strlen (wrBuffer), &retCnt)) < 0)
					return mxOX9000_status;
				break;
		case 1 :
				Fmt (wrBuffer, "ARM:COUP %s\r", cmdTrigCoupling[coupling]);
				if ((mxOX9000_status = viWrite (instrSession, wrBuffer, strlen (wrBuffer), &retCnt)) < 0)
					return mxOX9000_status;
				break;	
		case 2 :
				Fmt (wrBuffer, "ARM:FILT:HPAS 1\r");
				if ((mxOX9000_status = viWrite (instrSession, wrBuffer, strlen (wrBuffer), &retCnt)) < 0)
					return mxOX9000_status;
				break;	
		case 3 :
				Fmt (wrBuffer, "ARM:FILT:LPAS 1\r");
				if ((mxOX9000_status = viWrite (instrSession, wrBuffer, strlen (wrBuffer), &retCnt)) < 0)
					return mxOX9000_status;
				break;	
	}
	Fmt (wrBuffer, "ARM:HYST %i\r", noiseReject);
	if ((mxOX9000_status = viWrite (instrSession, wrBuffer, strlen (wrBuffer), &retCnt)) < 0)
		return mxOX9000_status;
		
	if(TrigMode == 0)
	{
		Fmt (wrBuffer ,"TRIG:RUN:STAT?\r");
		if ((mxOX9000_status = viPrintf (instrSession,wrBuffer,strlen (wrBuffer))) < 0)
			return mxOX9000_status;
		if ((mxOX9000_status = viScanf (instrSession,"%hd", &runStat)) < 0)
			return mxOX9000_status;
		Fmt (wrBuffer, "INIT:CONT:NAME DEL,%i[b1]\r", runStat);
		if ((mxOX9000_status = viWrite (instrSession, wrBuffer, strlen (wrBuffer), &retCnt)) < 0)
			return mxOX9000_status;
	}
	 else if(TrigMode == 2)
 	{
		Fmt (wrBuffer, "INIT:NAME DEL\r");
		if ((mxOX9000_status = viWrite (instrSession, wrBuffer, strlen (wrBuffer), &retCnt)) < 0)
			return mxOX9000_status; 
 	}
	
	/*-Check instrument status & Return from function------------------------*/
	if ((mxOX9000_status = mxOX9000_instrStatus(instrSession)) < 0)
		return mxOX9000_status;

	return mxOX9000_status;
}

/*===========================================================================*/
/* Function: Count Trigger                                                   */
/* Purpose:  This function sets event-count trigger type                     */
/*===========================================================================*/
ViStatus _VI_FUNC mxOX9000_setCountTrigger (ViSession instrSession, ViInt16 auxi_source, ViBoolean edge, ViInt16 coupling,
											ViInt16 noiseReject, ViReal64 level, ViInt32 events)
{                                
	ViStatus mxOX9000_status = VI_SUCCESS;
	ViByte wrBuffer[BUFFER_SIZE];
	ViInt32 retCnt;
	ViByte runStat = 1;

	/*-Check input parameter ranges------------------------------------------*/
	if (mxOX9000_invalidViInt16Range (auxi_source, 1, 4))
		return VI_ERROR_PARAMETER2;
	if (mxOX9000_invalidViBooleanRange (edge))
		return VI_ERROR_PARAMETER3;
	if (mxOX9000_invalidViInt16Range (coupling, 0, 3))
		return VI_ERROR_PARAMETER4;
	if (mxOX9000_invalidViInt16Range (noiseReject, 0, 3))
		return VI_ERROR_PARAMETER5;
	if (mxOX9000_invalidViReal64Range (level, -800, 800))
		return VI_ERROR_PARAMETER5;
	if (mxOX9000_invalidViInt32Range (events, 3, 16400))
		return VI_ERROR_PARAMETER6;

	/*-Set count trigger parameters------------------------------------------*/
	if ((mxOX9000_status = viPrintf (instrSession,"ARM:SOUR INT%hd\r", auxi_source)) < 0)
		return mxOX9000_status;

	if ((mxOX9000_status = viPrintf (instrSession,"ARM:SLOP %s\r", cmdSlope[edge])) < 0)
		return mxOX9000_status;

	if ((mxOX9000_status = viPrintf (instrSession,"ARM:LEV %lf\r", level)) < 0)
		return mxOX9000_status;

	if ((mxOX9000_status = viPrintf (instrSession,"TRIG:ECO %ld\r",events)) < 0)
		return mxOX9000_status;

	switch (coupling)
	{
		case 0 :
				Fmt (wrBuffer, "ARM:COUP %s\r", cmdTrigCoupling[coupling]);
				if ((mxOX9000_status = viWrite (instrSession, wrBuffer, strlen (wrBuffer), &retCnt)) < 0)
					return mxOX9000_status;
				break;
		case 1 :
				Fmt (wrBuffer, "ARM:COUP %s\r", cmdTrigCoupling[coupling]);
				if ((mxOX9000_status = viWrite (instrSession, wrBuffer, strlen (wrBuffer), &retCnt)) < 0)
					return mxOX9000_status;
				break;	
		case 2 :
				Fmt (wrBuffer, "ARM:FILT:HPAS 1\r");
				if ((mxOX9000_status = viWrite (instrSession, wrBuffer, strlen (wrBuffer), &retCnt)) < 0)
					return mxOX9000_status;
				break;	
		case 3 :
				Fmt (wrBuffer, "ARM:FILT:LPAS 1\r");
				if ((mxOX9000_status = viWrite (instrSession, wrBuffer, strlen (wrBuffer), &retCnt)) < 0)
					return mxOX9000_status;
				break;	
	}
	
	Fmt (wrBuffer, "ARM:HYST %i\r", noiseReject);
	if ((mxOX9000_status = viWrite (instrSession, wrBuffer, strlen (wrBuffer), &retCnt)) < 0)
		return mxOX9000_status;
		
	if(TrigMode == 0)
	{
		Fmt (wrBuffer ,"TRIG:RUN:STAT?\r");
		if ((mxOX9000_status = viPrintf (instrSession,wrBuffer,strlen (wrBuffer))) < 0)
			return mxOX9000_status;
		if ((mxOX9000_status = viScanf (instrSession,"%hd", &runStat)) < 0)
			return mxOX9000_status;
		Fmt (wrBuffer, "INIT:CONT:NAME EVEN,%i[b1]\r", runStat);
		if ((mxOX9000_status = viWrite (instrSession, wrBuffer, strlen (wrBuffer), &retCnt)) < 0)
			return mxOX9000_status;
	}
	else if(TrigMode == 2)
	{
		Fmt (wrBuffer, "INIT:NAME EVEN\r");
		if ((mxOX9000_status = viWrite (instrSession, wrBuffer, strlen (wrBuffer), &retCnt)) < 0)
			return mxOX9000_status; 
	}
	/*-Check instrument status & Return from function------------------------*/
	if ((mxOX9000_status = mxOX9000_instrStatus(instrSession)) < 0)
		return mxOX9000_status;

	return mxOX9000_status;
}

/*===========================================================================*/
/* Function: Display Options                                                 */
/* Purpose:  This function sets display options.                             */
/*===========================================================================*/
ViStatus _VI_FUNC mxOX9000_setOptionDisplay (ViSession instrSession, ViInt16 device_mode)
{
	ViStatus mxOX9000_status = VI_SUCCESS;
	ViByte wrBuffer[BUFFER_SIZE];
	ViInt32 retCnt;

	/*-Check input parameter ranges------------------------------------------*/
	if (mxOX9000_invalidViInt16Range(device_mode,0,4))
		return VI_ERROR_PARAMETER3;

	/*-Set option display parameters-----------------------------------------*/
	Fmt (wrBuffer, "DEV:MOD %s\r", cmddevice[device_mode]);
	if ((mxOX9000_status = viWrite (instrSession, wrBuffer, strlen (wrBuffer), &retCnt)) < 0)
		return mxOX9000_status;
	/*-Check instrument status & Return from function------------------------*/				   
	if ((mxOX9000_status = mxOX9000_instrStatus(instrSession)) < 0)
		return mxOX9000_status;
      
	return mxOX9000_status;
}

/*===========================================================================*/
/* Function: Display Waveform                                                */
/* Purpose:  This function sets waveform display.                            */
/*===========================================================================*/
ViStatus _VI_FUNC mxOX9000_setWaveformDisplay (ViSession instrSession, ViInt16 acquisition, ViInt16 average_rate,
											   ViInt16 mode, ViBoolean repet)
{
	ViStatus mxOX9000_status = VI_SUCCESS;
	ViByte wrBuffer[BUFFER_SIZE];
	ViInt32 retCnt;

	/*-Check input parameter ranges------------------------------------------*/
	if (mxOX9000_invalidViInt16Range (acquisition, 0, 1))
		return VI_ERROR_PARAMETER2;
	if (mxOX9000_invalidViInt16Range (average_rate, 0, 4))
		return VI_ERROR_PARAMETER3;
	if (mxOX9000_invalidViInt16Range(mode, 0, 3))
		return VI_ERROR_PARAMETER4;
	if (mxOX9000_invalidViBooleanRange (repet))
		return VI_ERROR_PARAMETER5;

	/*-Set waveform display parameters---------------------------------------*/
	Fmt (wrBuffer, "AVER:TYPE %s\r", cmdMaxminAcq[acquisition]);
	if ((mxOX9000_status = viWrite (instrSession, wrBuffer, strlen (wrBuffer), &retCnt)) < 0)
		return mxOX9000_status;
	Fmt (wrBuffer, "AVER %i[b2]\r", repet);
	if ((mxOX9000_status = viWrite (instrSession, wrBuffer, strlen (wrBuffer), &retCnt)) < 0)
		return mxOX9000_status;
	Fmt (wrBuffer, "DISP:TRAC:MODE %s\r", cmdMode[mode]);
	if ((mxOX9000_status = viWrite (instrSession, wrBuffer, strlen (wrBuffer), &retCnt)) < 0)
		return mxOX9000_status;
	if (repet == 1)
	{
		Fmt (wrBuffer, "AVER:COUN %s\r", cmdAverage[average_rate]);
		if ((mxOX9000_status = viWrite (instrSession, wrBuffer, strlen (wrBuffer), &retCnt)) < 0)
			return mxOX9000_status;
    }
	
    /*-Check instrument status & Return from function------------------------*/
	if ((mxOX9000_status = mxOX9000_instrStatus(instrSession)) < 0)
		return mxOX9000_status;
        
	return mxOX9000_status;
}

/*===========================================================================*/
/* Function: Set Cursors                                                     */
/* Purpose:  This function sets mxOX9000 cursors.                            */
/*===========================================================================*/
ViStatus _VI_FUNC mxOX9000_setCursors (ViSession instrSession, ViInt16 cursor, ViInt16 reference,
									   ViReal64 t1, ViReal64 t2, ViReal64 tphase)
{                              
	ViStatus mxOX9000_status = VI_SUCCESS;
	ViByte wrBuffer[BUFFER_SIZE];
	ViInt32 retCnt;
    
	/*-Check input parameter ranges------------------------------------------*/
	if (mxOX9000_invalidViInt16Range (cursor, 0, 1))
		return VI_ERROR_PARAMETER2;
	if (mxOX9000_invalidViInt16Range (reference, 0, 3))
		return VI_ERROR_PARAMETER3;
	if (mxOX9000_invalidViReal64Range (t1, 0, 2000.0))
		return VI_ERROR_PARAMETER4;
	if (mxOX9000_invalidViReal64Range (t2, 0, 2000.0))
		return VI_ERROR_PARAMETER5;
	if (mxOX9000_invalidViReal64Range(tphase, 0, 2000.0))
		return VI_ERROR_PARAMETER6;    
	 
	/*-Set cursors-----------------------------------------------------------*/    
	Fmt (wrBuffer, "DISP:CURS:REF %s\r", cmdSource[reference]);
	if ((mxOX9000_status = viWrite (instrSession, wrBuffer, strlen (wrBuffer), &retCnt)) < 0)
		return mxOX9000_status;
    
	switch(cursor)
	{
		case 0:
			Fmt (wrBuffer, "DISP:CURS:STAT OFF\r");
			if ((mxOX9000_status = viWrite (instrSession, wrBuffer, strlen (wrBuffer), &retCnt)) < 0)
				return mxOX9000_status;
			break;
			
		case 1:
			Fmt (wrBuffer, "DISP:CURS:STAT ON\r");
			if ((mxOX9000_status = viWrite (instrSession, wrBuffer, strlen (wrBuffer), &retCnt)) < 0)
				return mxOX9000_status;
			Fmt (wrBuffer, "DISP:CURS:TIME1:POS %f\r", t1);
			if ((mxOX9000_status = viWrite (instrSession, wrBuffer, strlen (wrBuffer), &retCnt)) < 0)
				return mxOX9000_status;
			Fmt (wrBuffer, "DISP:CURS:TIME2:POS %f\r", t2);
			if ((mxOX9000_status = viWrite (instrSession, wrBuffer, strlen (wrBuffer), &retCnt)) < 0)
				return mxOX9000_status;
			Fmt (wrBuffer, "DISP:CURS:TIME3:POS %f\r", tphase);
			if ((mxOX9000_status = viWrite (instrSession, wrBuffer, strlen (wrBuffer), &retCnt)) < 0)
				return mxOX9000_status;	
			break;	
	}

	/*-Check instrument status & Return from function------------------------*/
	if ((mxOX9000_status = mxOX9000_instrStatus(instrSession)) < 0)
		return mxOX9000_status;
        
	return mxOX9000_status;
}

/*===========================================================================*/
/* Function: Run Autoset                                                     */
/* Purpose:  This function performs an automatic setup of the mxOX9000.      */
/*===========================================================================*/
ViStatus _VI_FUNC mxOX9000_runAutoset (ViSession instrSession)
{                              
	ViStatus mxOX9000_status = VI_SUCCESS;
	ViInt16 i,state;
	ViByte wrBuffer[BUFFER_SIZE];
	ViInt32 retCnt;
    
	Fmt (wrBuffer, "AUTOS:EXE\r");
	if ((mxOX9000_status = viWrite (instrSession, wrBuffer, strlen (wrBuffer), &retCnt)) < 0)
		return mxOX9000_status;

	/*-Check instrument status & Return from function------------------------*/
	if ((mxOX9000_status = mxOX9000_instrStatus(instrSession)) < 0)
		return mxOX9000_status;
        
	return mxOX9000_status;
}

/*===========================================================================*/
/* Function: Run Acquisition                                                 */
/* Purpose:  This function performs an acquisition session.                  */
/*===========================================================================*/
ViStatus _VI_FUNC mxOX9000_runAcquisition (ViSession instrSession, ViBoolean run)
{                              
	ViStatus mxOX9000_status = VI_SUCCESS;
	ViInt32 retCnt;
	ViByte rdBuffer[BUFFER_SIZE];
    
	/*-Check input parameter ranges------------------------------------------*/
	if (mxOX9000_invalidViBooleanRange (run))
		return VI_ERROR_PARAMETER3;
    
	/*-Get current trigger mode----------------------------------------------*/
	Fmt (rdBuffer, "TRIG:RUN:STAT %i[b2]\r", run);
	if ((mxOX9000_status = viWrite (instrSession, rdBuffer, strlen (rdBuffer), &retCnt)) < 0)
		return mxOX9000_status;
		
	/*-Check instrument status & Return from function------------------------*/
	if ((mxOX9000_status = mxOX9000_instrStatus(instrSession)) < 0)
		return mxOX9000_status;
        
    return mxOX9000_status;
}

/*===========================================================================*/
/* Function: Run Hardcopy                                                    */
/* Purpose:  This function performs a hardcopy.                              */
/*===========================================================================*/
ViStatus _VI_FUNC mxOX9000_runHardcopy (ViSession instrSession)
{                              
	ViStatus mxOX9000_status = VI_SUCCESS;
    
	if ((mxOX9000_status = viPrintf (instrSession,"HCOP:SDUM\r")) < 0)
		return mxOX9000_status;

	return mxOX9000_status;
}

/*===========================================================================*/
/* Function: Read Cursors                                                    */
/* Purpose:  This function reads mxOX9000 cursors.                           */
/*===========================================================================*/
ViStatus _VI_FUNC mxOX9000_readCursors (ViSession instrSession, ViInt16 reference, ViPReal64 t1, ViPReal64 t2, ViPReal64 dt,
										ViPReal64 tphase, ViPReal64 v1, ViPReal64 v2, ViPReal64 dv)
{                              
	ViStatus mxOX9000_status = VI_SUCCESS;
	ViByte rdBuffer[BUFFER_SIZE];
	ViInt32 retCnt;
	ViInt16 valtphase;

	/*-Check input parameter ranges------------------------------------------*/
	if (mxOX9000_invalidViInt16Range (reference, 0, 3))
		return VI_ERROR_PARAMETER2;

	/*-Display cursor menu---------------------------------------------------*/
	Fmt (rdBuffer, "DISP:CURS:REF %s\r", cmdSource[reference]);
	if ((mxOX9000_status = viWrite (instrSession, rdBuffer, strlen (rdBuffer), &retCnt)) < 0)
		return mxOX9000_status;
        
	/*-Set cursor positions--------------------------------------------------*/
	if ((mxOX9000_status = viPrintf (instrSession,"DISP:CURS:TIME1:POS?\r")) < 0)
		return mxOX9000_status;
	if ((mxOX9000_status = viRead (instrSession, rdBuffer, 20, &retCnt)) < 0)
		return mxOX9000_status;
	Scan (rdBuffer, "%f", t1);
	
	if ((mxOX9000_status = viPrintf (instrSession,"DISP:CURS:TIME2:POS?\r")) < 0)
		return mxOX9000_status;
	if ((mxOX9000_status = viRead (instrSession, rdBuffer, 20, &retCnt)) < 0)
		return mxOX9000_status;
	Scan (rdBuffer, "%f", t2);
	
	if ((mxOX9000_status = viPrintf (instrSession,"DISP:CURS:TIME3:POS?\r")) < 0)
		return mxOX9000_status;
	if ((mxOX9000_status = viRead (instrSession, rdBuffer, 20, &retCnt)) < 0)
		return mxOX9000_status;
	Scan (rdBuffer, "%f", tphase);
	*tphase = (*tphase - *t1) / (*t2 - *t1) * 360.0;

	if ((mxOX9000_status = viPrintf (instrSession,"DISP:CURS:VOLT1:POS?\r")) < 0)
		return mxOX9000_status;
	if ((mxOX9000_status = viRead (instrSession, rdBuffer, 20, &retCnt)) < 0)
		return mxOX9000_status;
	Scan (rdBuffer, "%f", v1);

	if ((mxOX9000_status = viPrintf (instrSession,"DISP:CURS:VOLT2:POS?\r")) < 0)
		return mxOX9000_status;
	if ((mxOX9000_status = viRead (instrSession, rdBuffer, 20, &retCnt)) < 0)
		return mxOX9000_status;
	Scan (rdBuffer, "%f", v2);
		
	/*-Calculate 'delta t' & 'delta v'---------------------------------------*/
	*dt = *t2 - *t1;
	*dv = *v2 - *v1;
        
	/*-Check instrument status & Return from function------------------------*/
	if ((mxOX9000_status = mxOX9000_instrStatus(instrSession)) < 0)
		return mxOX9000_status;
        
	return mxOX9000_status;
}

/*===========================================================================*/
/* Function: Read Measurement                                                */
/* Purpose:  This function reads a specified measurement.                    */
/*===========================================================================*/
ViStatus _VI_FUNC mxOX9000_readMeasurement (ViSession instrSession, ViInt16 measure, ViInt16 reference, 
											ViPReal64 reading)
{                              
	ViStatus mxOX9000_status = VI_SUCCESS;
	ViByte rdBuffer[BUFFER_SIZE];
	ViInt32 retCnt;

	/*-Check input parameter ranges------------------------------------------*/
	if (mxOX9000_invalidViInt16Range (measure, 0, 21))
		return VI_ERROR_PARAMETER2;
	if (mxOX9000_invalidViInt16Range (reference, 0, 3))
		return VI_ERROR_PARAMETER3;

	/*-Set hardcopy parameters-----------------------------------------------*/
	Fmt (rdBuffer, "DISP:CURS:REF %s\r", cmdSource[reference]);
	if ((mxOX9000_status = viWrite (instrSession, rdBuffer, strlen (rdBuffer), &retCnt)) < 0)
		return mxOX9000_status;

	// measure Vrms & Vrms_c
	if (measure == 6 || measure == 7)
		Fmt (rdBuffer, "MEAS:%s? %s,%s\r", cmdReadMeasure[measure], cmdSource[reference], cmdMeasRMS[measure == 7]);
	else
		Fmt (rdBuffer, "MEAS:%s? %s\r", cmdReadMeasure[measure], cmdSource[reference]);

	if ((mxOX9000_status = viWrite (instrSession, rdBuffer, strlen (rdBuffer), &retCnt)) < 0)
		return mxOX9000_status;
	if ((mxOX9000_status = viRead (instrSession, rdBuffer, BUFFER_SIZE, &retCnt)) < 0)
		return mxOX9000_status;
	if ((mxOX9000_status = Scan (rdBuffer,"%s>%f",reading))!=1)
		return VI_ERROR_INSTR_INTERPRETING_RESPONSE;
     
	/*-Check instrument status & Return from function------------------------*/
	if ((mxOX9000_status = mxOX9000_instrStatus(instrSession)) < 0)
		return mxOX9000_status;
        
	return mxOX9000_status;
}

/*===========================================================================*/
/* Function: Read Channel To Array                                           */
/* Purpose:  This function reads a trace from the instrument.                */
/*           Data is stored in an array                                      */
/*===========================================================================*/
ViStatus _VI_FUNC mxOX9000_readChToArr (ViSession instrSession, ViInt16 source,	ViReal64 _VI_FAR waveformArray[], 
										ViPInt32 numberOfSamples, ViPReal64 XStart, ViPReal64 XIncrement)
{   
	ViStatus mxOX9000_status = VI_SUCCESS;
	ViInt16 headerSize;
	ViInt32 retCnt,size, index, nb;
	ViInt16 tstRes;
	ViUInt16 interface;
	static ViByte tabByte[MAX_MEM_SIZE];
    
	/*-Check input parameter ranges------------------------------------------*/
	if (mxOX9000_invalidViInt16Range (source, 0, 3))
		return VI_ERROR_PARAMETER2;
    
	/*-Check if source is active---------------------------------------------*/
	if ((mxOX9000_status = viPrintf (instrSession,":DISP:TRAC:STAT%hd?\r", source + 1)) < 0)
		return mxOX9000_status;
	if ((mxOX9000_status = viRead (instrSession, tabByte, 5, &retCnt)) < 0)
		return mxOX9000_status;
	Scan (tabByte,"%i[b2]", &tstRes);
	if (tstRes != 1) 
		return VI_ERROR_mxOX9000_CHAN_NOT_ACTIVE;
    
	/*-Modify Time Out for big waveform transfers------------------------*/
	if ((mxOX9000_status = viSetAttribute (instrSession, VI_ATTR_TMO_VALUE, TMO_TRANSFER)) < 0)
		return mxOX9000_status;

	/*-Modify Protocol for binary file transfers-------------------------*/
	viGetAttribute (instrSession, VI_ATTR_INTF_TYPE, &interface);
	if (interface == VI_INTF_ASRL)
	{
		if ((mxOX9000_status = viSetAttribute (instrSession, VI_ATTR_ASRL_END_IN, VI_ASRL_END_NONE)) < 0)
			return mxOX9000_status;
	}

	/*-Set data format and start transfert-----------------------------------*/    
	if ((mxOX9000_status = viPrintf (instrSession,"TRAC? INT%hd\r", source + 1)) < 0)
		return mxOX9000_status;
	if (interface==VI_INTF_ASRL) 
	{
		// Read 11 characters (#an header) to define the length of the binary block following 
		if ((mxOX9000_status = viRead (instrSession, tabByte, HEADER_SIZE, &retCnt)) < 0)
			return mxOX9000_status;
		index = FindPattern (tabByte, 0, HEADER_SIZE, "#", 0, 0);
		headerSize = tabByte[index + 1] - 0x30;
		Scan(tabByte + index + 1, "%s[w1]>%i", &nb);
		Scan(tabByte + index + 2, "%s[w*]>%i[b4]", nb, &size);
		if ((mxOX9000_status = viRead (instrSession, tabByte + HEADER_SIZE, index + size + 7 + headerSize - HEADER_SIZE, &retCnt)) < 0)
			return mxOX9000_status;

		//-Reset Protocol for binary file transfers----------------------------
		if ((mxOX9000_status = viSetAttribute (instrSession, VI_ATTR_ASRL_END_IN, VI_ASRL_END_TERMCHAR)) < 0)
			return mxOX9000_status;
	}
	else if ((mxOX9000_status = viRead (instrSession, tabByte, MAX_MEM_SIZE, &retCnt)) < 0)
		return mxOX9000_status;
												   
	if (interface == VI_INTF_ASRL)
	{
		if ((mxOX9000_status = viSetAttribute (instrSession, VI_ATTR_ASRL_END_IN, VI_ASRL_END_TERMCHAR)) < 0)
			return mxOX9000_status;
	}
	
	/*-Reset Time Out ---------------------------------------------------*/
	if ((mxOX9000_status = viSetAttribute (instrSession, VI_ATTR_TMO_VALUE, TMO_VALUE)) < 0)
		return mxOX9000_status;
		
	/*-Make data readable----------------------------------------------------*/
	if ((mxOX9000_status = mxOX9000_makeDataReadable(tabByte,waveformArray,numberOfSamples,XStart,XIncrement)) < 0)
		return mxOX9000_status;
                            
	/*-Check instrument status & Return from function------------------------*/
	if ((mxOX9000_status = mxOX9000_instrStatus(instrSession)) < 0)
		return mxOX9000_status;
        
	return mxOX9000_status;
}

/*===========================================================================*/
/* Function: Read Channel To File                                            */
/* Purpose:  This function reads a trace from the instrument.                */
/*           Data is stored in a file                                        */
/*===========================================================================*/
ViStatus _VI_FUNC mxOX9000_readChToFile (ViSession instrSession, ViInt16 channel, ViInt16 drive, ViString filename, ViBoolean overwrite)
{                        
	ViStatus mxOX9000_status = VI_SUCCESS;
	ViInt16 headerSize;
	ViInt32 fileSize, retCnt, size, index;													
	static ViByte tabByte[MAX_MEM_SIZE];
	ViByte wrBuffer[BUFFER_SIZE];
	ViInt16 tstRes;
	ViString filenameTemp = "___tmp__.trc";

	/*-Check input parameter ranges------------------------------------------*/
	if (mxOX9000_invalidViInt16Range (channel, 0, 3))
		return VI_ERROR_PARAMETER2;								 
	if (mxOX9000_invalidViInt16Range (drive, 0, 1))
		return VI_ERROR_PARAMETER3;
	if (mxOX9000_invalidViBooleanRange (overwrite))
		return VI_ERROR_PARAMETER5;
        
	/*-Check if source is active---------------------------------------------*/
	if ((mxOX9000_status = viPrintf (instrSession,":DISP:TRAC:STAT%hd?\r", channel + 1)) < 0)
		return mxOX9000_status;
	if ((mxOX9000_status = viRead (instrSession, tabByte, 5, &retCnt)) < 0)
		return mxOX9000_status;
	Scan (tabByte,"%i[b2]", &tstRes);
	if (tstRes != 1) 
		return VI_ERROR_mxOX9000_CHAN_NOT_ACTIVE;
    
	/*-Check filename--------------------------------------------------------*/
	if (mxOX9000_invalidFilename (filename, drive, ".trc"))
		return VI_ERROR_PARAMETER4;
    
	// Go to local memory and "traces" directory
	if ((mxOX9000_status = viPrintf (instrSession,"MMEM:MSIS LOCAL\r")) < 0)
		return mxOX9000_status;
	if ((mxOX9000_status = viPrintf (instrSession,"MMEM:CDIR \"/traces\"\r")) < 0)
		return mxOX9000_status;
	
	/*-Store current trace into mxOX9000---------------------------------------*/
	if (drive) 
	{
		/*-Check overwrite---------------------------------------------------*/
		if ((mxOX9000_status = viPrintf (instrSession,"MMEM:CAT?\r")) < 0)
			return mxOX9000_status;
		if ((mxOX9000_status = viRead (instrSession, tabByte, MAX_MEM_SIZE, &retCnt)) < 0)
			return mxOX9000_status;
		if (FindPattern (tabByte, 0, -1, filename, 0, 0) != -1) 
		{
			if (overwrite) 
			{
				if ((mxOX9000_status = viPrintf (instrSession,"MMEM:DEL \"%s\"\r", filename)) < 0)
					return mxOX9000_status;
			}
			else 
				return VI_ERROR_mxOX9000_EXISTING_FILE;
		}
        
		/*-Perform transfer--------------------------------------------------*/
		Fmt (wrBuffer, "MMEM:STOR:TRAC \"%s\",LOCAL\r", filename);
		if ((mxOX9000_status = viWrite (instrSession, wrBuffer, NumFmtdBytes (), &retCnt)) < 0)
			return mxOX9000_status;

		mxOX9000_status = viSetAttribute (instrSession, VI_ATTR_TMO_VALUE, TMO_TRANSFER);
		Fmt (wrBuffer, "*OPC?\r");
		mxOX9000_status = viWrite (instrSession, wrBuffer, NumFmtdBytes (), &retCnt);
		mxOX9000_status = viRead (instrSession, wrBuffer, BUFFER_SIZE, &retCnt);
		mxOX9000_status = viSetAttribute (instrSession, VI_ATTR_TMO_VALUE, TMO_VALUE);
	}
    
	/*-Store current trace into a PC file------------------------------------*/
	else 
	{
		/*-Check overwrite---------------------------------------------------*/
		if (GetFileInfo(filename,&fileSize)  && !overwrite) 
			return VI_ERROR_mxOX9000_EXISTING_FILE;

		if (GetFileInfo (filename, &fileSize)  && overwrite) 
			DeleteFile (filename);

		/*-Perform transfer--------------------------------------------------*/
		Fmt (wrBuffer, "MMEM:STOR:TRAC \"%s\",LOCAL\r", filenameTemp);
		if ((mxOX9000_status = viWrite (instrSession, wrBuffer, NumFmtdBytes (), &retCnt)) < 0)
			return mxOX9000_status;

		mxOX9000_status = viSetAttribute (instrSession, VI_ATTR_TMO_VALUE, TMO_TRANSFER);
		Fmt (wrBuffer, "*OPC?\r");
		mxOX9000_status = viWrite (instrSession, wrBuffer, NumFmtdBytes (), &retCnt);
		mxOX9000_status = viRead (instrSession, wrBuffer, BUFFER_SIZE, &retCnt);
		mxOX9000_status = viSetAttribute (instrSession, VI_ATTR_TMO_VALUE, TMO_VALUE);

		/*-Temp file transfer------------------------------------------------*/
		if ((mxOX9000_status = mxOX9000_fileTransfer (instrSession, filenameTemp, 1, VI_ON)) < 0)
			return mxOX9000_status;

		/*-Rename Temp file--------------------------------------------------*/
		RenameFile (filenameTemp, filename);
			
		/*-Delete Temp file--------------------------------------------------*/
		if ((mxOX9000_status = viPrintf (instrSession,"MMEM:DEL \"%s\"\r", filenameTemp)) < 0)
			return mxOX9000_status;
	}

	/*-Check instrument status & Return from function------------------------*/
	if ((mxOX9000_status = mxOX9000_instrStatus(instrSession)) < 0)
		return mxOX9000_status;
        
	return mxOX9000_status;
}

/*===========================================================================*/
/* Function: Write File To Instrument Display Memory                         */
/* Purpose:  This function fills the instrument display memory.              */
/*           Data are in the specified file.                                 */
/*===========================================================================*/
ViStatus _VI_FUNC mxOX9000_writeFileToMem (ViSession instrSession, ViString filename)
{   
	ViStatus mxOX9000_status = VI_SUCCESS;
	ViUInt32 retCnt;
	static ViByte tabByte[MAX_MEM_SIZE];
	ViByte wrBuffer[BUFFER_SIZE];

	/*-Check filename--------------------------------------------------------*/
	if (mxOX9000_invalidFilename (filename,1,".TRC"))
		return VI_ERROR_PARAMETER3;
    
	// Go to local memory and "traces" directory
	if ((mxOX9000_status = viPrintf (instrSession,"MMEM:MSIS LOCAL\r")) < 0)
		return mxOX9000_status;
	if ((mxOX9000_status = viPrintf (instrSession,"MMEM:CDIR \"/traces\"\r")) < 0)
		return mxOX9000_status;
	
	/*-Check filename--------------------------------------------------------*/
	if ((mxOX9000_status = viPrintf (instrSession,"MMEM:CAT?\r")) < 0)
		return mxOX9000_status;
	if ((mxOX9000_status = viRead (instrSession, tabByte, MAX_MEM_SIZE, &retCnt)) < 0)
		return mxOX9000_status;
	if (FindPattern (tabByte, 0, -1, filename , 0, 0) < 0)
		return VI_ERROR_mxOX9000_FILE_NOT_FOUND;

	/*-Perform transfer------------------------------------------------------*/
	Fmt (wrBuffer, "MMEM:LOAD:TRAC \"%s\",LOCAL\r", filename);
	if ((mxOX9000_status = viWrite (instrSession, wrBuffer, NumFmtdBytes (), &retCnt)) < 0)
		return mxOX9000_status;
		
	/*-Check instrument status & Return from function------------------------*/
	if ((mxOX9000_status = mxOX9000_instrStatus(instrSession)) < 0)
		return mxOX9000_status;
    
	return mxOX9000_status;
}

/*===========================================================================*/
/* Function: Save/Recall Setup From/To Instrument                            */
/* Purpose:  This function allows you to save the current setup of the       */
/*           instrument or to recall a setup stored in a file.               */
/*===========================================================================*/
ViStatus _VI_FUNC mxOX9000_saveSetup (ViSession instrSession, ViBoolean mode, ViInt16 drive, ViString filename, ViBoolean overwrite)
{
	ViStatus mxOX9000_status = VI_SUCCESS;
	ViInt16 headerSize;
	ViUInt32 retCnt, fileSize;
	static ViByte string[2000], tabByte[MAX_MEM_SIZE];
	ViUInt16 interface;

	/*-Check input parameter ranges------------------------------------------*/
	if (mxOX9000_invalidViBooleanRange (mode))
		return VI_ERROR_PARAMETER2;
	if (mxOX9000_invalidViInt16Range (drive, 0, 1))
		return VI_ERROR_PARAMETER3;
	if (mxOX9000_invalidViBooleanRange (overwrite))
		return VI_ERROR_PARAMETER5;

	/*-Check filename--------------------------------------------------------*/
	if (mxOX9000_invalidFilename (filename,drive,".CFG"))
		return VI_ERROR_PARAMETER4;

	// Go to local memory and "traces" directory
	if ((mxOX9000_status = viPrintf (instrSession,"MMEM:MSIS LOCAL\r")) < 0)
		return mxOX9000_status;
	if ((mxOX9000_status = viPrintf (instrSession,"MMEM:CDIR \"/setups\"\r")) < 0)
		return mxOX9000_status;
	
	/*-Save setup from mxOX9000----------------------------------------------*/
	if (mode) 	
	{
		/*-Target drive is mxOX9000------------------------------------------*/
		if (drive) 
		{
			/*-Check overwrite-----------------------------------------------*/
			if ((mxOX9000_status = viPrintf (instrSession,"MMEM:CAT? LOCAL\r")) < 0)
				return mxOX9000_status;
			if ((mxOX9000_status = viRead (instrSession, tabByte, MAX_MEM_SIZE, &retCnt)) < 0)
				return mxOX9000_status;
			if (FindPattern (tabByte, 0, -1, filename, 0, 0) != -1) 
			{
				if (overwrite) 
				{
					if ((mxOX9000_status = viPrintf (instrSession, "MMEM:DEL \"%s\"\r", filename)) < 0)
						return mxOX9000_status;
				}
                else return VI_ERROR_mxOX9000_EXISTING_FILE;
			}
			/*-Perform transfer----------------------------------------------*/
			if ((mxOX9000_status = viPrintf (instrSession, "MMEM:STORE:STATE \"%s\"\r",filename)) < 0)
				return mxOX9000_status;
		}
		/*-Target drive is PC------------------------------------------------*/
		else 
		{
			/*-Check overwrite-----------------------------------------------*/    
			if (GetFileInfo(filename,&fileSize)  && !overwrite) 
				return VI_ERROR_mxOX9000_EXISTING_FILE;
            
			/*-Modify Protocol for binary file transfers---------------------*/
			viGetAttribute (instrSession, VI_ATTR_INTF_TYPE, &interface);
			if(interface==VI_INTF_ASRL)
			{
				if ((mxOX9000_status = viSetAttribute (instrSession, VI_ATTR_ASRL_END_IN, VI_ASRL_END_NONE)) < 0)
					return mxOX9000_status;
			}
			/*-Perform transfer----------------------------------------------*/    
			if ((mxOX9000_status = viPrintf (instrSession,"SYST:SET?\r")) < 0)
				return mxOX9000_status;
            
			if (interface==VI_INTF_ASRL) 
			{
				/* Read 11 characters (#an header) to define the length of the binary block following */
				if ((mxOX9000_status = viRead (instrSession, tabByte, HEADER_SIZE, &retCnt)) < 0)
				    return mxOX9000_status;

				headerSize = tabByte[1] - 0x30;
				Scan(tabByte,"%s[i2]>%i[b4]",&fileSize);
				if ((mxOX9000_status = viRead (instrSession, tabByte + HEADER_SIZE, fileSize+4+headerSize-HEADER_SIZE, &retCnt)) < 0)
				     return mxOX9000_status;

				retCnt += HEADER_SIZE;

				/*-Reset Protocol for binary file transfers------------------*/
				if ((mxOX9000_status = viSetAttribute (instrSession, VI_ATTR_ASRL_END_IN, VI_ASRL_END_TERMCHAR)) < 0)
				    return mxOX9000_status;
			}
			else if ((mxOX9000_status = viRead (instrSession, tabByte, MAX_MEM_SIZE, &retCnt)) < 0)
				return mxOX9000_status;
            
			/*-Fill PC file--------------------------------------------------*/    
			if ((ArrayToFile (filename, tabByte, VAL_UNSIGNED_CHAR, retCnt - 2,retCnt - 2, VAL_GROUPS_TOGETHER,
					VAL_GROUPS_AS_ROWS,VAL_CONST_WIDTH, 4, VAL_BINARY,VAL_TRUNCATE)) < 0) 
				return VI_ERROR_INSTR_FILE_WRITE;
		}
	}
    
	/*-Recall setup to mxOX9000----------------------------------------------*/
	else 
	{
		/*-Source drive is mxOX9000------------------------------------------*/
		if (drive) 
		{
			/*-Check file name-----------------------------------------------*/    
			if ((mxOX9000_status = viPrintf (instrSession,"MMEM:CAT? LOCAL\r")) < 0)
				return mxOX9000_status;
			if ((mxOX9000_status = viRead (instrSession, tabByte, MAX_MEM_SIZE,&retCnt)) < 0)
				return mxOX9000_status;
			if (FindPattern (tabByte, 0, -1, filename, 0, 0) < 0)
				return VI_ERROR_mxOX9000_FILE_NOT_FOUND;
				
			/*-Perform transfer----------------------------------------------*/
			if ((mxOX9000_status = viPrintf (instrSession,"MMEM:LOAD:STATE \"%s\"\r",filename)) < 0)
				return mxOX9000_status;
		}
		/*-Source drive is PC------------------------------------------------*/
		else 
		{
			/*-Check file name-----------------------------------------------*/
			if (!GetFileInfo(filename,&fileSize)) 
				return VI_ERROR_mxOX9000_FILE_NOT_FOUND;
			/*-Read PC file--------------------------------------------------*/
			if (FileToArray (filename, tabByte, VAL_UNSIGNED_CHAR, 20, 20,VAL_GROUPS_TOGETHER,VAL_GROUPS_AS_ROWS, VAL_BINARY) < 0)
				return VI_ERROR_INSTR_FILE_OPEN;
			Scan(tabByte,"%s[i2]>%i[b4]",&fileSize);
			headerSize = 6;
			if (FileToArray (filename, tabByte, VAL_UNSIGNED_CHAR, headerSize + fileSize, 
					headerSize + fileSize, VAL_GROUPS_TOGETHER,VAL_GROUPS_AS_ROWS, VAL_BINARY) < 0)
				return VI_ERROR_INSTR_FILE_OPEN;
			/*-Perform transfer----------------------------------------------*/
			Fmt(string,"%s<SYST:SET %s[t-w*q]\r", headerSize + fileSize, tabByte);
			if ((mxOX9000_status = viWrite (instrSession, string, NumFmtdBytes (), &retCnt)) < 0) 
				return mxOX9000_status;
		}																  
		Delay(1.5);	
	}
    
	/*-Check instrument status & Return from function------------------------*/
	if ((mxOX9000_status = mxOX9000_instrStatus(instrSession)) < 0)
		return mxOX9000_status;
	 
	return mxOX9000_status;
}

/*===========================================================================*/
/* Function: File Transfer                                                   */ 
/* Purpose:  This function allows you to transfer a file from a drive        */
/*           to another.                                                     */
/*===========================================================================*/
ViStatus _VI_FUNC mxOX9000_fileTransfer (ViSession instrSession, ViString filename, ViInt16 srcDrive, ViBoolean overwrite)
{
	ViStatus mxOX9000_status = VI_SUCCESS;
	ViByte trueFilename[MAX_FILENAME_LEN];
	ViInt16 headerSize;
	ViInt32 retCnt,fileSize, size, index;
	static ViByte tabByte[MAX_MEM_SIZE], tempString[20];
	ViByte wrBuffer[BUFFER_SIZE];
	ViUInt16 interface;

	/*-Check input parameter ranges------------------------------------------*/
	if (mxOX9000_invalidViInt16Range (srcDrive, 0, 1))
		return VI_ERROR_PARAMETER3;
	if (mxOX9000_invalidViBooleanRange (overwrite))
		return VI_ERROR_PARAMETER4;
  
	/*-Check filename--------------------------------------------------------*/
	if ((FindPattern (filename, 0, -1, ".", 0, 0) > 28) && (strlen (filename) > 32))
		return VI_ERROR_PARAMETER2;

	// Go to local memory and "traces" directory
	if ((mxOX9000_status = viPrintf (instrSession,"MMEM:MSIS LOCAL\r")) < 0)
		return mxOX9000_status;
	if (FindPattern (filename, 0, -1, ".trc", 1, 0) != -1)
		Fmt (wrBuffer, "MMEM:CDIR \"/traces\"\r");
	else if (FindPattern (filename, 0, -1, ".cfg", 1, 0) != -1)
		Fmt (wrBuffer, "MMEM:CDIR \"/setups\"\r");
	else if (FindPattern (filename, 0, -1, ".png", 1, 0) != -1)
		Fmt (wrBuffer, "MMEM:CDIR \"/screenshots\"\r");
	else if (FindPattern (filename, 0, -1, ".rec", 1, 0) != -1)
		Fmt (wrBuffer, "MMEM:CDIR \"/logger\"\r");
	else if (FindPattern (filename, 0, -1, ".htm", 1, 0) != -1)
		Fmt (wrBuffer, "MMEM:CDIR \"/bus\"\r");
	else
		Fmt (wrBuffer, "MMEM:CDIR \"/\"\r");
	
	if ((mxOX9000_status = viWrite (instrSession, wrBuffer, strlen (wrBuffer), &retCnt)) < 0)
		return mxOX9000_status;
	
	/*-Computer to mxOX9000 transfer-----------------------------------------*/
	if (!srcDrive ) 
	{
		/*-Check file name---------------------------------------------------*/    
		if (!GetFileInfo(filename,&fileSize)) 
			return VI_ERROR_mxOX9000_FILE_NOT_FOUND;
		SplitPath (filename, NULL, NULL, trueFilename);

		/*-Check overwrite---------------------------------------------------*/    
		if ((mxOX9000_status = viPrintf (instrSession, "MMEM:CAT? LOCAL\r")) < 0)
			return mxOX9000_status;
		Delay(1.5);	
		if ((mxOX9000_status = viRead (instrSession, tabByte, MAX_MEM_SIZE, &retCnt)) < 0)
			return mxOX9000_status;
		if (FindPattern (tabByte, 0, -1, filename, 0, 0) != -1) 
		{
			if (overwrite) 
			{
				if ((mxOX9000_status = viPrintf (instrSession, "MMEM:DEL \"%s\"\r", filename)) < 0)
					return mxOX9000_status;
			}
			else   
				return VI_ERROR_mxOX9000_EXISTING_FILE;
		}
    
		/*-Read PC file------------------------------------------------------*/
		GetFileInfo (filename, &fileSize);
     
		/*-Read any file but not ".trc" or ".cfg"----------------------------*/
		if ((FindPattern(trueFilename, 0, -1, "trc", 0, 0) == -1) && 
			(FindPattern(trueFilename, 0, -1, "cfg", 0, 0) == -1)) 
		{
			/*-Perform transfer----------------------------------------------*/
			Fmt (tempString, "%i[b4]", fileSize);
			Fmt (tabByte, "MMEM:DATA \"%s\", #%i[b2]%i[b4]", trueFilename, strlen (tempString), fileSize);
		}
        
		/*-Read ".TRC" or ".CFG" files---------------------------------------*/
		else 
		{
			/*-Perform transfer----------------------------------------------*/
			Fmt (tabByte, "MMEM:DATA \"%s\",", trueFilename);
		}
			
		headerSize = strlen (tabByte);

		if (FileToArray (filename, tabByte + headerSize, VAL_UNSIGNED_CHAR, fileSize, 1,
		                	VAL_GROUPS_TOGETHER, VAL_GROUPS_AS_ROWS, VAL_BINARY) < 0)
			return VI_ERROR_INSTR_FILE_OPEN;

		tabByte[headerSize + fileSize] = '\r';
		
		if ((mxOX9000_status = viWrite (instrSession, tabByte, headerSize + fileSize + 1, &retCnt)) < 0) 
			return mxOX9000_status;
		
		Delay(1.5);	
	}
    
	/*-mxOX9000 to Computer transfer-----------------------------------------*/
	else 
	{
		/*-Check filename----------------------------------------------------*/
		if ((mxOX9000_status = viPrintf (instrSession, "MMEM:CAT? LOCAL\r")) < 0)
			return mxOX9000_status;
		if ((mxOX9000_status = viRead (instrSession, tabByte, MAX_MEM_SIZE, &retCnt)) < 0)
			return mxOX9000_status;
		if (FindPattern (tabByte, 0, -1, filename, 0, 0) < 0)
			return VI_ERROR_mxOX9000_FILE_NOT_FOUND;

		/*-Check overwrite---------------------------------------------------*/    
		if (GetFileInfo(filename, &fileSize)  && !overwrite) 
			return VI_ERROR_mxOX9000_EXISTING_FILE;

		/*-Modify Time Out for big file transfers----------------------------*/
		if ((mxOX9000_status = viSetAttribute (instrSession, VI_ATTR_TMO_VALUE, TMO_TRANSFER)) < 0)
			return mxOX9000_status;

		/*-Modify Protocol for binary file transfers-------------------------*/
		viGetAttribute (instrSession, VI_ATTR_INTF_TYPE, &interface);
		if(interface==VI_INTF_ASRL)
		{
			if ((mxOX9000_status = viSetAttribute (instrSession, VI_ATTR_ASRL_END_IN, VI_ASRL_END_NONE)) < 0)
				return mxOX9000_status;
		}

		/*-Perform transfer--------------------------------------------------*/
		if ((mxOX9000_status = viPrintf (instrSession, "MMEM:DATA? \"%s\"\r", filename)) < 0)
			return mxOX9000_status;

		if (interface==VI_INTF_ASRL) 
       	{
			/* Read 11 characters (#an header) to define the length of the binary block following */
			if ((mxOX9000_status = viRead (instrSession, tabByte, HEADER_FILE, &retCnt)) < 0)
				return mxOX9000_status;

			headerSize = tabByte[1] - 0x30;
			Scan(tabByte, "%s[i2]>%i[b4]", &fileSize);
			if ((mxOX9000_status = viRead (instrSession, tabByte+HEADER_FILE, fileSize+4+headerSize-HEADER_FILE, &retCnt)) < 0)
				return mxOX9000_status;

			retCnt += HEADER_FILE;

			/*-Reset Protocol for binary file transfers----------------------*/
			if ((mxOX9000_status = viSetAttribute (instrSession, VI_ATTR_ASRL_END_IN, VI_ASRL_END_TERMCHAR)) < 0)
				return mxOX9000_status;
		}
		else if ((mxOX9000_status = viRead (instrSession, tabByte, MAX_MEM_SIZE, &retCnt)) < 0)
			return mxOX9000_status;

		/*-Reset Time Out ---------------------------------------------------*/
		if ((mxOX9000_status = viSetAttribute (instrSession, VI_ATTR_TMO_VALUE, TMO_VALUE)) < 0)
			return mxOX9000_status;

		/*-Parse non ".TRC" or ".CFG" files----------------------------------*/
		if ((FindPattern(filename,0,-1,"trc",0,0)==-1) && (FindPattern(filename, 0, -1, "cfg", 0, 0)==-1))
			headerSize = tabByte[1] - 0x30 + 2; 
		else
			headerSize = 0;

		/*-Fill PC file------------------------------------------------------*/    
		if ((ArrayToFile (filename, tabByte + headerSize, VAL_UNSIGNED_CHAR, retCnt - headerSize - 2, 1, 
		VAL_GROUPS_TOGETHER, VAL_GROUPS_AS_ROWS, VAL_CONST_WIDTH, 4, VAL_BINARY, VAL_TRUNCATE)) < 0) 
			return VI_ERROR_INSTR_FILE_WRITE;
	}
	
	/*-Check instrument status & Return from function------------------------*/
	if ((mxOX9000_status = mxOX9000_instrStatus (instrSession)) < 0)
		return mxOX9000_status;

	return mxOX9000_status;
}

/*===========================================================================*/
/* Function: Write To Instrument                                             */
/* Purpose:  This function writes a command string to the instrument.        */
/*===========================================================================*/
ViStatus _VI_FUNC mxOX9000_writeInstrData (ViSession instrSession, ViString writeBuffer)
{
	ViStatus mxOX9000_status = VI_SUCCESS;

	if ((mxOX9000_status = viPrintf (instrSession, "%s", writeBuffer)) < 0)
		return mxOX9000_status;

	return mxOX9000_status;
}

/*===========================================================================*/
/* Function: Read Instrument Buffer                                          */
/* Purpose:  This function reads the output buffer of the instrument.        */
/*===========================================================================*/
ViStatus _VI_FUNC mxOX9000_readInstrData (ViSession instrSession, ViInt32 numBytes,
							ViChar _VI_FAR rdBuf[], ViPInt32 bytesRead)

{
	ViStatus mxOX9000_status = VI_SUCCESS;
	*bytesRead = 0L;

	if ((mxOX9000_status = viRead (instrSession, rdBuf, BUFFER_SIZE, bytesRead)) < 0)
		return mxOX9000_status;

	return mxOX9000_status;
}

/*===========================================================================*/
/* Function: Reset                                                           */
/* Purpose:  This function resets the instrument.                            */
/*===========================================================================*/
ViStatus _VI_FUNC  mxOX9000_reset (ViSession instrSession)
{
	ViUInt32 retCnt = 0;
	ViStatus  mxOX9000_status = VI_SUCCESS;

	/*-Initialize the instrument to a known state----------------------------*/
	if (( mxOX9000_status = viWrite (instrSession, "\r*RST\r", 6, &retCnt)) < 0)
		return  mxOX9000_status;

	if (( mxOX9000_status =  mxOX9000_defaultInstrSetup (instrSession)) < 0)  
		return  mxOX9000_status;
        
	return  mxOX9000_status;
}

/*===========================================================================*/
/* Function: Self-Test                                                       */
/* Purpose:  This function executes the instrument self-test and returns     */
/*           the result.                                                     */
/*===========================================================================*/
ViStatus _VI_FUNC mxOX9000_selfTest (ViSession instrSession, ViPInt16 testResult, ViChar _VI_FAR testMessage[])
{
	ViStatus mxOX9000_status = VI_SUCCESS;
	ViUInt32 retCnt = 0;
	ViByte rdBuffer[BUFFER_SIZE];

	if ((mxOX9000_status = viSetAttribute (instrSession, VI_ATTR_TMO_VALUE,20 * TMO_VALUE)) < 0)
		return mxOX9000_status; 
    
	if ((mxOX9000_status = viWrite (instrSession, "*TST?\r", 6, &retCnt)) < 0)
		return mxOX9000_status;
	if ((mxOX9000_status = viRead (instrSession, rdBuffer, 50, &retCnt)) < 0)
		return mxOX9000_status;
	if (Scan (rdBuffer, "%i[b1]", testResult) != 1)
		return VI_ERROR_INSTR_INTERPRETING_RESPONSE;

	if (*testResult) 
		Fmt(testMessage, "%s<Self test failed");
	else 
		Fmt(testMessage, "%s<No Error");  

	if ((mxOX9000_status = viSetAttribute (instrSession, VI_ATTR_TMO_VALUE, 
		TMO_VALUE)) < 0)
		return mxOX9000_status;

	return mxOX9000_status;
}

/*===========================================================================*/
/* Function: Error Query                                                     */
/* Purpose:  This function queries the instrument error queue, and returns   */
/*           the result. If the error query function is not supported by the */
/*           instrument, this function returns the warning                   */
/*           VI_WARN_NSUP_ERROR_QUERY.                                       */
/*===========================================================================*/
ViStatus _VI_FUNC mxOX9000_errorQuery (ViSession instrSession, ViPInt32 errCode, ViChar _VI_FAR errMessage[])
{
	ViUInt32 retCnt = 0;
	ViStatus mxOX9000_status = VI_SUCCESS;
	ViByte rdBuffer[BUFFER_SIZE];

	if ((mxOX9000_status = viWrite (instrSession, "SYST:ERR?\r", 10, &retCnt)) < 0)
		return mxOX9000_status;
	if ((mxOX9000_status = viRead (instrSession, rdBuffer, BUFFER_SIZE, &retCnt)) < 0)
		return mxOX9000_status;

	if (Scan (rdBuffer, "%i[b4],%s[t10]", errCode,errMessage) != 2)
		return VI_ERROR_INSTR_INTERPRETING_RESPONSE;

	return mxOX9000_status;
}

/*===========================================================================*/
/* Function: Error Message                                                   */
/* Purpose:  This function translates the error return value from the        */
/*           instrument driver into a user-readable string.                  */
/*===========================================================================*/
ViStatus _VI_FUNC mxOX9000_errorMessage (ViSession instrSession, ViStatus errorCode, ViChar _VI_FAR errMessage[])
{
	ViStatus mxOX9000_status = VI_SUCCESS;
	ViInt16 i;
    
	static mxOX9000_tStringValPair statusDescArray[] = {
		{VI_ERROR_PARAMETER1,   "ERROR: Parameter 1 out of range"},
		{VI_ERROR_PARAMETER2,   "ERROR: Parameter 2 out of range"},
		{VI_ERROR_PARAMETER3,   "ERROR: Parameter 3 out of range"},
		{VI_ERROR_PARAMETER4,   "ERROR: Parameter 4 out of range"},
		{VI_ERROR_PARAMETER5,   "ERROR: Parameter 5 out of range"},
		{VI_ERROR_PARAMETER6,   "ERROR: Parameter 6 out of range"},
		{VI_ERROR_PARAMETER7,   "ERROR: Parameter 7 out of range"},
		{VI_ERROR_PARAMETER8,   "ERROR: Parameter 8 out of range"},
		{VI_ERROR_INSTR_PARAMETER9,   "ERROR: Parameter 9 out of range"},
		{VI_ERROR_FAIL_ID_QUERY,"ERROR: Identification query failed"},
		{VI_ERROR_INSTR_FILE_OPEN,    "ERROR: Opening the specified file"},
		{VI_ERROR_INSTR_FILE_WRITE,   "ERROR: Writing to the specified file"},
		{VI_ERROR_INSTR_INTERPRETING_RESPONSE,           "ERROR: Interpreting the instrument's response"},
		{VI_ERROR_mxOX9000_ERROR,                  "ERROR: Instrument specific error"},
		{VI_ERROR_mxOX9000_INCONSISTENT_PARAMETER3,"ERROR: Parameter 3 is INCONSISTENT"},
		{VI_ERROR_mxOX9000_INCONSISTENT_PARAMETER5,"ERROR: Parameter 5 is INCONSISTENT"},
		{VI_ERROR_mxOX9000_INCONSISTENT_PARAMETER6,"ERROR: Parameter 6 is INCONSISTENT"},
		{VI_ERROR_mxOX9000_INCONSISTENT_PARAMETER7,"ERROR: Parameter 7 is INCONSISTENT"},
		{VI_ERROR_mxOX9000_AUTOTEST_PROBLEM,       "ERROR: Performing a self-test"}, 
		{VI_ERROR_mxOX9000_CHAN_NOT_ACTIVE,        "ERROR: Channel is not active"},
		{VI_ERROR_mxOX9000_NO_CHAN_ENABLED,        "ERROR: No channel is enabled"},
		{VI_ERROR_mxOX9000_FILE_NOT_FOUND,         "ERROR: File not available on support"},
		{VI_ERROR_mxOX9000_EXISTING_FILE,          "ERROR: Filename already used"},
		{VI_ERROR_mxOX9000_INVALID_FILE_CONTENT,   "ERROR: File content is invalid"},
		{VI_NULL, VI_NULL}  };
	          
	mxOX9000_status = viStatusDesc (instrSession, errorCode, errMessage);
	if (mxOX9000_status == VI_WARN_UNKNOWN_STATUS) 
	{
		for (i=0; statusDescArray[i].stringName; i++) 
		{
			if (statusDescArray[i].stringVal == errorCode) 
			{
				Fmt (errMessage, "%s<%s", statusDescArray[i].stringName);
				return (VI_SUCCESS);
			}
		}
		Fmt (errMessage, "%s<Unknown Error 0x%x[uw8p0]", errorCode);
		return (VI_WARN_UNKNOWN_STATUS);
	}
	mxOX9000_status = VI_SUCCESS;
	return mxOX9000_status;
}

/*===========================================================================*/
/* Function: Revision Query                                                  */
/* Purpose:  This function returns the driver and instrument revisions.      */
/*===========================================================================*/
ViStatus _VI_FUNC mxOX9000_revisionQuery (ViSession instrSession, ViChar _VI_FAR driverRev[], ViChar _VI_FAR instrRev[])
{
	ViUInt32 retCnt = 0;
	ViStatus mxOX9000_status = VI_SUCCESS;
	ViByte rdBuffer[BUFFER_SIZE];

	if ((mxOX9000_status = viWrite (instrSession, "*IDN?\r", 6, &retCnt)) < 0)
		return mxOX9000_status;
	if ((mxOX9000_status = viRead (instrSession, rdBuffer, BUFFER_SIZE, &retCnt)) < 0)
		return mxOX9000_status;
	if (Scan (rdBuffer, "%s[t10]", instrRev) != 1)
		return VI_ERROR_INSTR_INTERPRETING_RESPONSE;
	Fmt (driverRev, "%s<%s", mxOX9000_REVISION);

	return mxOX9000_status;
}

/*===========================================================================*/
/* Function: MeterConfig                                                     */
/* Purpose:  This function configures the Meter mode                         */
/*===========================================================================*/
ViStatus _VI_FUNC mxOX9000_Meterconfig (ViSession instrSession, ViInt16 input, ViInt16 measure, ViInt16 coupling, ViInt16 bandwith,  
										ViBoolean autoRange, ViInt16 VoltRang, ViInt16 OhmRang, ViInt16 CapaRang)
{
	ViStatus mxOX9000_status = VI_SUCCESS;
	ViByte wrBuffer[BUFFER_SIZE];
	ViInt32 retCnt;
    
	/*-Check input parameter ranges------------------------------------------*/
	if (mxOX9000_invalidViInt16Range (input, 1, 4))
		return VI_ERROR_PARAMETER2;
	if (mxOX9000_invalidViInt16Range (measure, 0, 5))
		return VI_ERROR_PARAMETER3;
	if (mxOX9000_invalidViInt16Range (coupling, 0, 2))
		return VI_ERROR_PARAMETER4;
	if (mxOX9000_invalidViInt16Range (bandwith, 0, 2))
		return VI_ERROR_PARAMETER5;
	if ((mxOX9000_invalidViBooleanRange (autoRange))) 
		return VI_ERROR_PARAMETER6;
	if (mxOX9000_invalidViInt16Range (VoltRang, 0, 4))
		return VI_ERROR_PARAMETER7;
	if (mxOX9000_invalidViInt16Range (OhmRang, 0, 6))
		return VI_ERROR_PARAMETER8;
	if (mxOX9000_invalidViInt16Range (CapaRang, 0, 6))
		return VI_ERROR_PARAMETER9;
		
	/*-Set the meter parameters----------------------------------------------*/
	Fmt (wrBuffer, "FUNC %s\r", CmdMultMeas[measure]);
	if ((mxOX9000_status = viWrite (instrSession, wrBuffer, strlen (wrBuffer), &retCnt)) < 0)
		return mxOX9000_status;
		
	Fmt (wrBuffer, "RANG%i:AUTO %i[b2]\r", input, autoRange);
	if ((mxOX9000_status = viWrite (instrSession, wrBuffer, strlen (wrBuffer), &retCnt)) < 0)
		return mxOX9000_status;
	
	// coupling in voltmeter
	if (measure == 0)
	{
		Fmt (wrBuffer, "INP%i:DMM:COUP %s\r", input, MulCoup[coupling]);
		if ((mxOX9000_status = viWrite (instrSession, wrBuffer, strlen (wrBuffer), &retCnt)) < 0)
			return mxOX9000_status;
		
		if (coupling != 0)
		{
			Fmt (wrBuffer, "INP%i:DMM:BAND:RES %f\r", input, MulBand[bandwith]);
			if ((mxOX9000_status = viWrite (instrSession, wrBuffer, strlen (wrBuffer), &retCnt)) < 0)
				return mxOX9000_status;
		}
	}
	
	if (!autoRange)
	{
		switch (measure)
		{
			case 0:
				Fmt (wrBuffer, "RANG%i:VOLT %s\r", input, CmdMultVolt[VoltRang]);
				if ((mxOX9000_status = viWrite (instrSession, wrBuffer, strlen (wrBuffer), &retCnt)) < 0)
					return mxOX9000_status;	
				break;

			case 1:
				Fmt (wrBuffer, "RANG:OHM %s\r", CmdMultOhm[OhmRang]);
				if ((mxOX9000_status = viWrite (instrSession, wrBuffer, strlen (wrBuffer), &retCnt)) < 0)
					return mxOX9000_status;	
				break;
	
			case 3:
				Fmt (wrBuffer, "RANG:CAPA %s\r", CmdMulCapa[CapaRang]);
				if ((mxOX9000_status = viWrite (instrSession, wrBuffer, strlen (wrBuffer), &retCnt)) < 0)
					return mxOX9000_status;	
				break;
		}
	}
	/*-Check instrument status & Return from function------------------------*/
	if ((mxOX9000_status = mxOX9000_instrStatus(instrSession)) < 0)
		return mxOX9000_status;

	return mxOX9000_status;
}	

/*===========================================================================*/
/* Function: Read MeterMeas                                                  */
/* Purpose:  This function reads the main measurement to the meter mode      */
/*===========================================================================*/
ViStatus _VI_FUNC mxOX9000_readMeterMeas (ViSession instrSession, ViInt16 ref, ViPReal64 reading)
{
	ViStatus mxOX9000_status = VI_SUCCESS;
	ViByte rdBuffer[BUFFER_SIZE];
	ViInt32 retCnt;
	ViByte mult;
	
	/*-Check input parameter ranges------------------------------------------*/
	if (mxOX9000_invalidViInt16Range (ref, 1, 4))
		return VI_ERROR_PARAMETER2;

	/*-Set measure parameter-------------------------------------------------*/
	if ((mxOX9000_status = viPrintf (instrSession,"MEAS:DMM? INT%i\r", ref)) < 0)
		return mxOX9000_status;
	if ((mxOX9000_status = viScanf (instrSession, "%lf%c", reading, &mult)) < 0)
		return mxOX9000_status;		

	switch (mult)
	{
		case 'G':  *reading *= 1e+9;  break;
		case 'M':  *reading *= 1e+6;  break;
		case 'k':  *reading *= 1e+3;  break;
		case 'm':  *reading *= 1e-3;  break;
		case 0xB5: *reading *= 1e-6;  break;
		case 'p':  *reading *= 1e-12; break;
	}
		
	/*-Check instrument status & Return from function------------------------*/
	if ((mxOX9000_status = mxOX9000_instrStatus(instrSession)) < 0)
		return mxOX9000_status;

	return mxOX9000_status;
}

/*===========================================================================*/
/* Function: Close                                                           */
/* Purpose:  This function closes the instrument.                            */
/*===========================================================================*/
ViStatus _VI_FUNC  mxOX9000_close (ViSession instrSession)
{

	mxOX9000_instrRange instrPtr;
	ViSession rmSession;
	ViStatus  mxOX9000_status = VI_SUCCESS;

	if (( mxOX9000_status = viGetAttribute (instrSession, VI_ATTR_RM_SESSION, &rmSession)) < 0)
		return  mxOX9000_status;
	if (( mxOX9000_status = viGetAttribute (instrSession, VI_ATTR_USER_DATA, &instrPtr)) < 0)
		return  mxOX9000_status;
    
	if(instrPtr != NULL)
		free (instrPtr);
    
	mxOX9000_status = viClose (instrSession);
	viClose (rmSession);

	return  mxOX9000_status;
}

/*===========================================================================*/
/* Function: Boolean Value Out Of Range - ViBoolean                          */
/* Purpose:  This function checks a Boolean to see if it is equal to VI_TRUE */
/*           or VI_FALSE. If the value is out of range, the return value is  */
/*           VI_TRUE, otherwise the return value is VI_FALSE.                */
/*===========================================================================*/
ViBoolean  mxOX9000_invalidViBooleanRange (ViBoolean val)
{
	return ((val != VI_FALSE && val != VI_TRUE) ? VI_TRUE : VI_FALSE);
}

/*===========================================================================*/
/* Function: Short Signed Integer Value Out Of Range - ViInt16               */
/* Purpose:  This function checks a short signed integer value to see if it  */  
/*           lies between a minimum and maximum value.  If the value is out  */
/*           of range, the return value is VI_TRUE, otherwise the return     */
/*           value is VI_FALSE.                                              */
/*===========================================================================*/
ViBoolean  mxOX9000_invalidViInt16Range (ViInt16 val, ViInt16 min, ViInt16 max)
{
	return ((val < min || val > max) ? VI_TRUE : VI_FALSE);
}

/*===========================================================================*/
/* Function: Long Signed Integer Value Out Of Range - ViInt32                */
/* Purpose:  This function checks a long signed integer value to see if it   */  
/*           lies between a minimum and maximum value.  If the value is out  */
/*           of range,  the return value is VI_TRUE, otherwise the return    */
/*           value is VI_FALSE.                                              */
/*===========================================================================*/
ViBoolean  mxOX9000_invalidViInt32Range  (ViInt32 val, ViInt32 min, ViInt32 max)
{
	return ((val < min || val > max) ? VI_TRUE : VI_FALSE);
}

/*===========================================================================*/
/* Function: Short Unsigned Integer Value Out Of Range - ViUInt16            */
/* Purpose:  This function checks a short unsigned integer value to see if it*/  
/*           lies between a minimum and maximum value.  If the value is out  */
/*           of range,  the return value is VI_TRUE, otherwise the return    */
/*           value is VI_FALSE.                                              */
/*===========================================================================*/
ViBoolean  mxOX9000_invalidViUInt16Range  (ViUInt16 val, ViUInt16 min, ViUInt16 max)
{
	return ((val < min || val > max) ? VI_TRUE : VI_FALSE);
}

/*===========================================================================*/
/* Function: Long Unsigned Integer Value Out Of Range - ViUInt32             */
/* Purpose:  This function checks a long unsigned integer value to see if it */  
/*           lies between a minimum and maximum value.  If the value is out  */
/*           of range,  the return value is VI_TRUE, otherwise the return    */
/*           value is VI_FALSE.                                              */
/*===========================================================================*/
ViBoolean  mxOX9000_invalidViUInt32Range  (ViUInt32 val, ViUInt32 min, ViUInt32 max)
{
	return ((val < min || val > max) ? VI_TRUE : VI_FALSE);
}

/*===========================================================================*/
/* Function: Real (Float) Value Out Of Range - ViReal32                      */
/* Purpose:  This function checks a real (float) value to see if it lies     */  
/*           between a minimum and maximum value.  If the value is out of    */
/*           range, the return value is VI_TRUE, otherwise the return value  */
/*           is VI_FALSE.                                                    */
/*===========================================================================*/
ViBoolean  mxOX9000_invalidViReal32Range  (ViReal32 val, ViReal32 min, ViReal32 max)
{
	return ((val < min || val > max) ? VI_TRUE : VI_FALSE);
}

/*===========================================================================*/
/* Function: Real (Double) Value Out Of Range - ViReal64                     */
/* Purpose:  This function checks a real (double) value to see if it lies    */  
/*           between a minimum and maximum value.  If the value is out of    */
/*           range, the return value is VI_TRUE, otherwise the return value  */
/*           is VI_FALSE.                                                    */
/*===========================================================================*/
ViBoolean  mxOX9000_invalidViReal64Range  (ViReal64 val, ViReal64 min, ViReal64 max)
{
	return ((val < min || val > max) ? VI_TRUE : VI_FALSE);
}

/*===========================================================================*/
/* Function: Instrument Status                                               */
/* Purpose:  This function checks the instrument error status, by reading    */
/*           and processing instrument's event status register. An error     */
/*           is reported if bits 4 and 5 resp. EXE and CME are asserted.     */
/*===========================================================================*/
ViStatus  mxOX9000_instrStatus (ViSession instrSession)
{
	ViUInt32 retCnt = 0;
	ViStatus  mxOX9000_status = VI_SUCCESS;
	ViByte rdBuffer[BUFFER_SIZE];
	ViByte status;
    
	/*- Command to query Event Status Register ------------------------------*/
	if (( mxOX9000_status = viPrintf (instrSession,"*ESR?\r")) < 0)
		return  mxOX9000_status;
        
	if (( mxOX9000_status = viRead (instrSession, rdBuffer, 50, &retCnt)) < 0)
		return  mxOX9000_status;

	/*- Scanning the response -----------------------------------------------*/
	if (Scan (rdBuffer, "%i[b1]", &status) != 1)
		return VI_ERROR_INSTR_INTERPRETING_RESPONSE;
    
	/*- Testing bits 4 and 5 ------------------------------------------------*/
	if (status & 0x30) 
		return VI_ERROR_mxOX9000_ERROR;
    
	return  mxOX9000_status;
}

/*===========================================================================*/
/* Function: Initialize Clean Up                                             */
/* Purpose:  This function is used only by the  mxOX9000_init function.  When*/
/*           an error is detected this function is called to close the       */
/*           open resource manager and instrument object sessions and to     */
/*           set the instrSession that is returned from  mxOX9000_init to    */
/*           VI_NULL.                                                        */
/*===========================================================================*/
ViStatus  mxOX9000_initCleanUp (ViSession openRMSession,ViPSession openInstrSession, ViStatus currentStatus)
{
	viClose (*openInstrSession);
	viClose (openRMSession);
	*openInstrSession = VI_NULL;
    
	return currentStatus;
}

/*===========================================================================*/
/* Function: Default Instrument Setup                                        */
/* Purpose:  This function sends a default setup to the instrument.  This    */
/*           function is called by the  mxOX9000_reset operation and by the  */
/*            mxOX9000_init function if the reset option has not been        */
/*           selected.  This function is useful for configuring any          */
/*           instrument settings that are required by the rest of the        */
/*           instrument driver functions such as turning headers ON or OFF   */
/*           or using the long or short form for commands, queries, and data.*/                                    
/*===========================================================================*/
ViStatus  mxOX9000_defaultInstrSetup (ViSession instrSession)
{
	ViStatus  mxOX9000_status = VI_SUCCESS;
	ViUInt32 retCnt = 0;
	mxOX9000_instrRange instrPtr;
        
	/* Determine if the structure has been initialized for the current VISA  */
	/* Session and malloc if it has not.                                     */
	if ( mxOX9000_status = viGetAttribute (instrSession, VI_ATTR_USER_DATA, &instrPtr))
		return  mxOX9000_status;
    
	if (instrPtr == NULL) 
		instrPtr = malloc (sizeof (struct  mxOX9000_statusDataRanges));
    
	instrPtr -> triggerMode = 0;
	instrPtr -> val2 = 0;
	instrPtr -> val3 = 0;
	Fmt (instrPtr -> instrDriverRevision, "%s<%s",  mxOX9000_REVISION);
    
	if (( mxOX9000_status = viSetAttribute (instrSession, VI_ATTR_USER_DATA, (ViUInt32)instrPtr)) < 0)
		return  mxOX9000_status;                                       

	if (( mxOX9000_status = viWrite (instrSession, "*CLS;:FORM INT;:FORM:DINT OFF\r", 30, &retCnt)) < 0)
		return  mxOX9000_status;

	return  mxOX9000_status;    
}

/*===========================================================================*/
/* Function: Make data readable                                              */
/* Purpose:  This function fills output parameters with readable data        */
/*===========================================================================*/
ViStatus mxOX9000_makeDataReadable  (ViByte tabByte[], ViReal64 _VI_FAR waveformArray[], ViPInt32 numberOfSamples, 
									 ViPReal64 XStart, ViPReal64 XIncrement)
{
	ViStatus mxOX9000_status = VI_SUCCESS;
	ViInt32 i, index, start, type = 0, tempo;
	ViReal64 Offset, YIncrement = 0; 
	union 
	{
		ViByte  buffer[4];
		ViInt32 valSample; 
	} Buff;

#define OFFSET_ADC	393216		//0x60000 en hexa

	/*-Find begining of data file----------------------------------------*/    
	start = FindPattern (tabByte, 0, 200, "#64000", 0, 0);
	if (start == -1) 
		return VI_ERROR_mxOX9000_INVALID_FILE_CONTENT;
	
	start += 8;
    index = FindPattern (tabByte, 0, 60, "SCAL", 0, 0);
	Scan (tabByte + index + 5, "%f", XIncrement);

	index = FindPattern (tabByte, 0, 90, "SIZE", 0, 0);
	Scan (tabByte + index + 5, "%i[b4]", numberOfSamples);
	 
	index = FindPattern (tabByte, 0, 200, "OFFS", 0, 0);
	Scan (tabByte + index + 5, "%f", &Offset);

	index = FindPattern (tabByte, 80, 200, "SCAL", 0, 0);
	Scan (tabByte + index + 5, "%f", &YIncrement);

	for (i=0; i<*numberOfSamples; i++)
	{
		tempo = *(long *)(tabByte + start + i * 4) & 0x000FFFFF;
		waveformArray[i] = (tempo - OFFSET_ADC) * YIncrement + (Offset + 12 * YIncrement * 0x8000);
	}
	
	return mxOX9000_status;
}

/*===========================================================================*/
/* Function: Check File Name                                                 */
/* Purpose:  This function checks if the file name is properly edited        */		
/*===========================================================================*/
ViBoolean mxOX9000_invalidFilename (ViString filename, ViInt16 drive, ViString type)
{
	ViInt16 startIndex = 0;
    
	if (!drive) 
		startIndex = FindPattern (filename, 0, -1, "\\", 0, 1) + 1;
	if (strlen (filename + startIndex) > 12) 
		return VI_TRUE;
	startIndex = FindPattern (filename, 0, -1, ".", 0, 1);
	if (CompareStrings (filename, startIndex, type, 0, 0) != 0)  
		return VI_TRUE; 
		
	return VI_FALSE;
}

